/*******************************************************************************
 * Copyright (c) 2011 Michael Ruflin, Andr Locher, Claudia von Bastian.
 * 
 * This file is part of Tatool.
 * 
 * Tatool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published 
 * by the Free Software Foundation, either version 3 of the License, or 
 * (at your option) any later version.
 * 
 * Tatool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with Tatool. If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ch.tatool.app.export;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import au.com.bytecode.opencsv.CSVWriter;
import ch.tatool.data.DataService;
import ch.tatool.data.Module;
import ch.tatool.data.DataContainer.Entry;

/**
 * Exports training data into a CSV file.
 * This is a helper class used by the various exporters
 * 
 * @author Andre Locher
 */
public class CSVModuleDataExport {

    Logger logger = LoggerFactory.getLogger(CSVModuleDataExport.class);

    private List<String> baseProperties;
    private int basePropertiesSize;
    private List<String> moduleProperties;
    private int sessionPropertiesSize;
    
    public CSVModuleDataExport(DataService dataService) {
        // this won't change
        initBaseProperties();
    }
    
    /**
     * Exports the module data and stores the results in the provided Writer
     * 
     * @param module
     * @param file the file to write the data to
     * @return a file containing the module data or null in case an error occured
     */
    public File exportData(Module module) {
        // initialize the module properties
    	initModuleProperties(module);

        // create a writer
        try {
            File tmpFile = File.createTempFile("moduleData", "csv");
            FileOutputStream fos = new FileOutputStream(tmpFile, false);
            CSVWriter writer = new CSVWriter(new OutputStreamWriter(fos,"ISO-8859-1"), ';');
  
            // write the csv file header
            List<String> headers = new ArrayList<String>(baseProperties);
            headers.addAll(moduleProperties);
            writer.writeNext(headers.toArray(new String[headers.size()]));
   
            // fetch module data
            List<String[]> moduleData = getDataForModule(module);
            writer.writeAll(moduleData);
            
            // close the writer and return the file
            writer.close();
            return tmpFile;
        } catch (IOException ioe) {
            logger.error("Unable to write csv file", ioe);
            return null;
        }
    }
    
    /** Returns a list with all to be written values for a given trial object. */
    private List<String[]> getDataForModule(Module module) {
        List<String[]> data = new ArrayList<String[]>(basePropertiesSize + sessionPropertiesSize);

        // add module properties
        addModuleProperties(module, data);
        
        return data;
    }
    
    // Base properties management
    
    /** Basic properties which are always written. */
    private void initBaseProperties() {
        baseProperties = new ArrayList<String>();
        
        // Module related
        baseProperties.add("Module id");
        baseProperties.add("Module name");

        basePropertiesSize = baseProperties.size();        
    }
    
    private String[] getBaseProperties(Module module) {
    	// module related
    	String[] baseProps = new String[2];
    	baseProps[0] = module.getId().toString();
    	baseProps[1] = module.getName();
    	return baseProps;
    }

    private void addToData(List<String[]> data, String[] values) {
        if (values != null) {
            data.add(values);
        }
    }

    // Module properties management

    private void initModuleProperties(Module module) {
        // add additional headers for module properties
    	moduleProperties = new ArrayList<String>();
    	moduleProperties.add("ELEMENT_ID");
    	moduleProperties.add("PROPERTY_NAME");
    	moduleProperties.add("PROPERTY_VALUE");
        sessionPropertiesSize = moduleProperties.size();   
    }

    /**
     * Adds the module properties to the data object
     */
    private void addModuleProperties(Module module, List<String[]> data) {
    	List<String> moduleProps = null;
    	
        for (Entry p : module.getEntries()) {
        	moduleProps = new ArrayList<String>();
        	moduleProps.addAll(Arrays.asList(getBaseProperties(module)));
        	moduleProps.add(p.getNodeId());
        	moduleProps.add(p.getName());
        	moduleProps.add(p.getValue());
        	addToData(data, moduleProps.toArray(new String[moduleProps.size()]));
        }
    }
}
