/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter;

import lombok.AccessLevel;
import lombok.Generated;
import lombok.experimental.FieldDefaults;

import java.util.function.Consumer;

/**
 *  A tree cursor is a stateful object that allows you to walk a syntax tree with maximum efficiency.
 */
@FieldDefaults(level = AccessLevel.PRIVATE)
public class TreeCursor extends External {

    int context0;
    int context1;

    long id;
    long tree;

    TreeCursor(Node node) {
        super(malloc(node));
    }

    static native long malloc(Node node);

    /**
     * Delete the tree cursor,
     * freeing all the memory that it used.
     */
    @Override
    public native void close();

    /**
     * @return The tree cursor's current node.
     */
    public native Node getCurrentNode();

    /**
     * @return The field name of the tree cursor's current node.
     * Will return null if the current node doesn't have a field.
     */
    public native String getCurrentFieldName();

    /**
     * @return The tree cursor's current node.
     */
    public native TreeCursorNode getCurrentTreeCursorNode();

    /**
     * Move the cursor to the first child of its current node.
     *
     * @return true if the cursor successfully moved, and false if there were no children.
     */
    public native boolean gotoFirstChild();

    /**
     * Move the cursor to the next sibling of its current node.
     *
     * @return true if the cursor successfully moved, and false if there was no next sibling node.
     */
    public native boolean gotoNextSibling();

    /**
     * Move the cursor to the parent of its current node.
     *
     * @return true if the cursor successfully moved, and false if there was no parent node
     * (the cursor was already on the root node).
     */
    public native boolean gotoParent();

    /**
     * Iteratively traverse over the parse tree, applying a callback to the nodes before they are visited.
     *
     * @param callback The callback consumer which will execute upon visiting a node.
     */
    public void preorderTraversal(Consumer<Node> callback) {
        for (;;) {
            callback.accept(this.getCurrentNode());
            if (this.gotoFirstChild() || this.gotoNextSibling())
                continue;
            do {
                if (!this.gotoParent())
                    return;
            } while (!this.gotoNextSibling());
        }
    }

    @Override
    @Generated
    public String toString() {
        return String.format("TreeCursor(id: %d, tree: %d)", id, tree);
    }
}
