/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.EqualsAndHashCode;
import lombok.Generated;
import lombok.Getter;
import lombok.experimental.Accessors;
import lombok.experimental.FieldDefaults;
import org.jetbrains.annotations.NotNull;

import java.util.Objects;

/**
 * Represents a two-dimensional point with row and column coordinates.
 * Points are an alternative to byte ranges, and as such are used to
 * represent more human-friendly positions of tree nodes within source code.
 * Although node positions within files should never be negative,
 * instances of this class can be created with negative row and column
 * values for other purposes, such as denoting repositioning offsets.
 *
 * @since 1.0.0
 * @author Ozren Dabić
 */
@Getter
@AllArgsConstructor
@EqualsAndHashCode
@FieldDefaults(level = AccessLevel.PRIVATE, makeFinal = true)
public class Point implements Comparable<Point> {

    @Getter(lazy = true)
    @Accessors(fluent = true, makeFinal = true)
    private static final Point ORIGIN = new Point(0, 0);

    int row;
    int column;

    /**
     * Returns a string representation of this point in the format:
     * <pre>{@code
     *      row:column
     * }</pre>
     *
     * @return a string representation of this point
     */
    @Override
    @Generated
    public String toString() {
        return row + ":" + column;
    }

    /**
     * Checks if this point represents the origin,
     * which is when both the row and the column are zero.
     * In byte range terms, this point also corresponds to zero.
     *
     * @return {@code true} if this is an origin point,
     * {@code false} otherwise
     */
    public boolean isOrigin() {
        return equals(ORIGIN());
    }

    /**
     * Compares this point with another point for positional order.
     * Points are compared by their row coordinates first,
     * and if those are equal they are then compared by their column coordinates.
     *
     * @param other the object to be compared
     * @return the row comparison result, or the column comparison result if the rows are equal
     * @throws NullPointerException if the other point is null
     * @since 1.5.1
     */
    @Override
    public native int compareTo(@NotNull Point other);

    /**
     * Adds another point to this point,
     * resulting in a new point with coordinates
     * equal to the sum of the coordinates
     * of this point and the other point.
     *
     * @param other the point to be added to this point
     * @return a new point
     * @throws NullPointerException if {@code other} is null
     * @since 1.5.1
     */
    public Point add(@NotNull Point other) {
        Objects.requireNonNull(other, "Other point must not be null!");
        if (isOrigin()) return other;
        if (other.isOrigin()) return this;
        if (equals(other)) return multiply(2);
        return new Point(row + other.row, column + other.column);
    }

    /**
     * Subtracts another point from this point,
     * resulting in a new point with coordinates
     * equal to the difference between the coordinates
     * of this point and the other point.
     *
     * @param other the point to be subtracted from this point
     * @return a new point
     * @throws NullPointerException if {@code other} is null
     * @since 1.5.1
     */
    public Point subtract(@NotNull Point other) {
        Objects.requireNonNull(other, "Other point must not be null!");
        if (isOrigin()) return other.multiply(-1);
        if (other.isOrigin()) return this;
        if (equals(other)) return ORIGIN();
        return new Point(row - other.row, column - other.column);
    }

    /**
     * Multiplies the coordinates of this point by a scalar value,
     * resulting in a new point with scaled coordinates.
     *
     * @param value the scalar value by which to multiply the coordinates of this point
     * @return a new point
     * @since 1.5.1
     */
    public Point multiply(int value) {
        if (value == 0) return ORIGIN();
        if (value == 1) return this;
        return new Point(row * value, column * value);
    }
}
