/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter.printer;

import ch.usi.si.seart.treesitter.TreeCursor;
import ch.usi.si.seart.treesitter.TreeCursorNode;
import org.jetbrains.annotations.NotNull;

import java.util.function.Consumer;

/**
 * Printer used for generating human-readable
 * representations of trees using an iterative approach.
 * Each node of the subtree is represented as follows:
 * <pre>{@code
 *      optional_field_name: node_name [start_column:start_row] - [end_column:end_row]
 * }</pre>
 * While indentation is used to represent the tree structure.
 *
 * @since 1.2.0
 * @see <a href="https://tree-sitter.github.io/tree-sitter/playground">Syntax Tree Playground</a>
 * @author Ozren Dabić
 */
public class SyntaxTreePrinter extends IterativeTreePrinter {

    public SyntaxTreePrinter(@NotNull TreeCursor cursor) {
        super(cursor);
    }

    @Override
    protected String getFileExtension() {
        return ".txt";
    }

    protected void write(Consumer<String> appender) {
        for (;;) {
            TreeCursorNode cursorNode = cursor.getCurrentTreeCursorNode();
            if (cursorNode.isNamed()) {
                int depth = cursor.getCurrentDepth();
                String indent = "  ".repeat(depth);
                appender.accept(indent);
                appender.accept(cursorNode.toString());
                appender.accept("\n");
            }
            if (cursor.gotoFirstChild() || cursor.gotoNextSibling()) continue;
            do {
                if (!cursor.gotoParent()) return;
            } while (!cursor.gotoNextSibling());
        }
    }
}
