/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Generated;
import lombok.Getter;
import lombok.experimental.FieldDefaults;

import java.util.Objects;

/**
 * Represents a symbol in an abstract syntax {@link Tree}.
 * Symbols are used to identify nodes in the AST.
 * Each symbol has an associated ID, {@link Type Type}, and name.
 *
 * @author Ozren Dabić
 * @since 1.6.0
 */
@Getter
@AllArgsConstructor(access = AccessLevel.PACKAGE)
@FieldDefaults(level = AccessLevel.PRIVATE, makeFinal = true)
public class Symbol {

    int id;
    Type type;
    String name;

    @SuppressWarnings("unused")
    Symbol(int id, int ordinal, String name) {
        this(id, Type.get(ordinal), name);
    }

    @Override
    @Generated
    public String toString() {
        return String.format("Symbol(id: %d, type: %s, name: '%s')", id, type, name);
    }

    @Override
    @Generated
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj == null || getClass() != obj.getClass()) return false;
        Symbol symbol = (Symbol) obj;
        return id == symbol.id && type == symbol.type && Objects.equals(name, symbol.name);
    }

    @Override
    @Generated
    public int hashCode() {
        return Objects.hash(id, type, name);
    }

    /**
     * Enumeration representing the possible types of symbols.
     * This includes:
     * <ul>
     *     <li>Named nodes ({@link #REGULAR})</li>
     *     <li>Anonymous nodes ({@link #ANONYMOUS})</li>
     *     <li>Hidden nodes ({@link #AUXILIARY})</li>
     * </ul>
     *
     * @author Ozren Dabić
     * @since 1.6.0
     */
    public enum Type {

        REGULAR,
        ANONYMOUS,
        AUXILIARY;

        private static final Type[] VALUES = values();

        private static Type get(int ordinal) {
            return VALUES[ordinal];
        }
    }
}
