/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter.function;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.function.Consumer;

/**
 * Represents an operation that accepts a single input argument and returns no result.
 * Can potentially throw an {@link IOException} while doing so.
 *
 * @since 1.2.0
 * @see java.util.function.Consumer Consumer
 * @param <T> the type of the input to the operation
 * @author Ozren Dabić
 */
@FunctionalInterface
public interface IOExceptionThrowingConsumer<T> {

    /**
     * Performs this operation on the given argument.
     *
     * @param t the input argument
     * @throws IOException if an I/O error occurs
     */
    void accept(T t) throws IOException;

    /**
     * Returns a wrapped {@code Consumer} that performs this operation.
     * If performing said operation results in an {@code IOException},
     * it is re-thrown as its unchecked counterpart: {@code UncheckedIOException}.
     *
     * @param consumer the operation to wrap
     * @param <T> the type of the input to the operation
     * @return and identical {@code Consumer} that throws
     * {@code UncheckedIOException} instead
     */
    static <T> Consumer<T> toUnchecked(IOExceptionThrowingConsumer<T> consumer) {
        return t -> {
            try {
                consumer.accept(t);
            } catch (IOException ex) {
                throw new UncheckedIOException(ex);
            }
        };
    }
}
