/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter.printer;

import ch.usi.si.seart.treesitter.Tree;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Objects;

/**
 * Printer used for generating DOT graph representations of trees.
 * Unlike its sister classes, it does not rely on an iterative approach,
 * relying instead on the internal {@code tree-sitter} API.
 *
 * @since 1.2.0
 * @author Ozren Dabić
 */
public class DotGraphPrinter implements TreePrinter {

    Tree tree;

    public DotGraphPrinter(Tree tree) {
        this.tree = Objects.requireNonNull(tree, "Tree must not be null!");
    }

    /**
     * Generates a DOT graph of the tree.
     *
     * @return A DOT graph string of the tree
     */
    @Override
    public String print() {
        try {
            File file = export();
            Path path = file.toPath();
            String contents = Files.readString(path);
            Files.delete(path);
            return contents;
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    /**
     * Generates a DOT graph of the tree,
     * writing it directly to a file.
     *
     * @return A file containing the DOT graph of the tree
     * @throws IOException if an I/O error occurs
     */
    @Override
    public File export() throws IOException {
        File file = Files.createTempFile("ts-export-", ".dot").toFile();
        write(file);
        return file;
    }

    native void write(File file) throws IOException;
}
