/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter;

import lombok.AccessLevel;
import lombok.Cleanup;
import lombok.RequiredArgsConstructor;
import lombok.experimental.FieldDefaults;

/**
 * Utility used for pretty-printing entire syntax trees, as well as their subtrees.
 *
 * @deprecated use the {@linkplain ch.usi.si.seart.treesitter.printer.SyntaxTreePrinter new implementation} instead.
 * @since 1.0.0
 * @author Ozren Dabić
 */
@Deprecated(since = "1.2.0", forRemoval = true)
@RequiredArgsConstructor
@FieldDefaults(level = AccessLevel.PRIVATE)
public class SyntaxTreePrinter {

    int depth = 0;
    final Node node;
    Point offset = new Point(0, 0);

    public SyntaxTreePrinter(Node node, Point offset) {
        this.node = node;
        this.offset = offset;
    }

    /**
     * @return A string representation of the subtree, consisting only of named nodes
     * @see <a href="https://tree-sitter.github.io/tree-sitter/playground">Syntax Tree Playground</a>
     */
    public String printSubtree() {
        StringBuilder stringBuilder = new StringBuilder();
        @Cleanup TreeCursor cursor = new TreePrinterCursor(node);
        for (;;) {
            TreeCursorNode cursorNode = cursor.getCurrentTreeCursorNode();
            if (cursorNode.isNamed()) {
                recalculatePosition(cursorNode);
                String indentation = "  ".repeat(depth);
                stringBuilder
                        .append(indentation)
                        .append(cursorNode)
                        .append("\n");
            }
            if (cursor.gotoFirstChild() || cursor.gotoNextSibling()) continue;
            do {
                if (!cursor.gotoParent()) return stringBuilder.toString();
            } while (!cursor.gotoNextSibling());
        }
    }

    private void recalculatePosition(TreeCursorNode cursorNode) {
        if (!offset.isOrigin()) {
            Point cursorStartPoint = cursorNode.getStartPoint();
            Point cursorEndPoint = cursorNode.getEndPoint();

            int rowOffset = offset.getRow();
            cursorStartPoint.setRow(cursorStartPoint.getRow() + rowOffset);
            cursorEndPoint.setRow(cursorEndPoint.getRow() + rowOffset);

            int columnOffset = offset.getColumn();
            cursorStartPoint.setColumn(cursorStartPoint.getColumn() + columnOffset);
            cursorEndPoint.setColumn(cursorEndPoint.getColumn() + columnOffset);
        }
    }

    private final class TreePrinterCursor extends TreeCursor {

        private TreePrinterCursor(Node node) {
            super(node);
        }

        @Override
        public boolean gotoFirstChild() {
            boolean success = super.gotoFirstChild();
            if (success) depth++;
            return success;
        }

        @Override
        public boolean gotoParent() {
            boolean success = super.gotoParent();
            if (success) depth--;
            return success;
        }
    }
}
