/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter.printer;

import ch.usi.si.seart.treesitter.TreeCursor;
import ch.usi.si.seart.treesitter.function.IOExceptionThrowingConsumer;
import lombok.AccessLevel;
import lombok.Cleanup;
import lombok.experimental.FieldDefaults;
import org.jetbrains.annotations.NotNull;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.file.Files;
import java.util.Objects;
import java.util.function.Consumer;

@FieldDefaults(level = AccessLevel.PROTECTED, makeFinal = true)
abstract class IterativeTreePrinter implements TreePrinter {

    TreeCursor cursor;

    protected IterativeTreePrinter(@NotNull TreeCursor cursor) {
        this.cursor = Objects.requireNonNull(cursor, "Cursor must not be null!");
    }

    /**
     * Generates a string representation of the Abstract Syntax Tree (AST),
     * starting from the node currently pointed to by the cursor.
     *
     * @return A string representation of the tree
     */
    public final String print() {
        StringBuilder stringBuilder = new StringBuilder(getPreamble());
        write(stringBuilder::append);
        return stringBuilder.toString();
    }

    /**
     * Generates a string representation of the Abstract Syntax Tree (AST),
     * starting from the node currently pointed to by the cursor,
     * while writing outputs directly to a file.
     *
     * @return A file containing the string of the tree
     * @throws IOException if an I/O error occurs
     */
    public final File export() throws IOException {
        File file = Files.createTempFile("ts-export-", getFileExtension()).toFile();
        @Cleanup Writer writer = new BufferedWriter(new FileWriter(file));
        writer.append(getPreamble());
        Consumer<String> appender = IOExceptionThrowingConsumer.toUnchecked(writer::append);
        try {
            write(appender);
        } catch (UncheckedIOException ex) {
            throw ex.getCause();
        }
        return file;
    }

    protected String getPreamble() {
        return "";
    }

    protected String getFileExtension() {
        return "";
    }

    protected abstract void write(Consumer<String> appender);
}
