/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter;

import lombok.AccessLevel;
import lombok.Generated;
import lombok.Getter;
import lombok.experimental.FieldDefaults;
import org.jetbrains.annotations.NotNull;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Iterator;

/**
 * A Tree represents the syntax tree of an entire source code file.
 * It contains {@link Node Node} instances that indicate the structure of the source code.
 *
 * @since 1.0.0
 * @author Tommy MacWilliam
 * @author Ozren Dabić
 */
@Getter
@FieldDefaults(level = AccessLevel.PRIVATE, makeFinal = true)
public class Tree extends External implements Iterable<Node> {

    private static final Charset CHARSET = StandardCharsets.UTF_16LE;

    Language language;
    String source;

    Tree(long pointer, @NotNull Language language, @NotNull String source) {
        super(pointer);
        this.language = language;
        this.source = source;
    }

    /**
     * Delete the tree, freeing all the memory that it used.
     */
    @Override
    public native void close();

    /**
     * Edit the syntax tree to keep it in sync with source code that has been edited.
     *
     * @param edit Changes made to the source code in terms of <em>both</em> byte offsets and row/column coordinates
     */
    public native void edit(@NotNull InputEdit edit);

    /**
     * @return The root node of the syntax tree
     */
    public native Node getRootNode();

    /**
     * @return An iterator over the entire syntax tree, starting from the root node
     */
    @Override
    public @NotNull Iterator<Node> iterator() {
        return getRootNode().iterator();
    }

    @Override
    @Generated
    public String toString() {
        return String.format("Tree(id: %d, language: %s)", pointer, language);
    }

    String getSource(int startByte, int endByte) {
        byte[] bytes = source.getBytes(CHARSET);
        byte[] copy = Arrays.copyOfRange(bytes, startByte * 2, endByte * 2);
        return new String(copy, CHARSET);
    }
}
