/*
 * MIT License
 *
 * Copyright (c) 2022-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.treesitter.printer;

import ch.usi.si.seart.treesitter.Point;
import ch.usi.si.seart.treesitter.TreeCursor;
import ch.usi.si.seart.treesitter.TreeCursorNode;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.function.Consumer;

/**
 * Printer used for generating Extensible Markup Language (XML)
 * representations of trees using an iterative approach.
 * Note that unlike the CLI counterpart, the resulting XML
 * document does not contain the actual source code contents.
 *
 * @since 1.2.0
 * @see <a href="https://github.com/tree-sitter/tree-sitter/blob/293f0d1ca30a63839810ad4b943c0f19f1cb4933/cli/src/parse.rs#L186-L239">Rust implementation</a>
 * @author Ozren Dabić
 */
public class XMLPrinter extends IterativeTreePrinter {

    public static final String PROLOG = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";

    public XMLPrinter(@NotNull TreeCursor cursor) {
        super(cursor);
    }

    @Override
    protected String getPreamble() {
        return PROLOG;
    }

    @Override
    protected String getFileExtension() {
        return ".xml";
    }

    protected void write(Consumer<String> appender) {
        boolean visitedChildren = false;
        Deque<String> tags = new ArrayDeque<>();
        for (;;) {
            TreeCursorNode cursorNode = cursor.getCurrentTreeCursorNode();
            boolean isNamed = cursorNode.isNamed();
            String type = cursorNode.getType();
            String name = cursorNode.getName();
            Point start = cursorNode.getStartPoint();
            Point end = cursorNode.getEndPoint();
            if (visitedChildren) {
                if (isNamed) {
                    appender.accept("</");
                    appender.accept(tags.pop());
                    appender.accept(">");
                }
                if (cursor.gotoNextSibling()) visitedChildren = false;
                else if (cursor.gotoParent()) visitedChildren = true;
                else return;
            } else {
                if (isNamed) {
                    appender.accept("<");
                    appender.accept(type);
                    appender.accept(" ");
                    if (name != null) {
                        appender.accept("name=\"");
                        appender.accept(name);
                        appender.accept("\" ");
                    }
                    appender.accept("start=\"");
                    appender.accept(start.toString());
                    appender.accept("\" ");
                    appender.accept("end=\"");
                    appender.accept(end.toString());
                    appender.accept("\">");
                    tags.push(type);
                }
                visitedChildren = !cursor.gotoFirstChild();
            }
        }
    }
}
