/*
 * MIT License
 *
 * Copyright (c) 2023-present SEART Research Group and Contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ch.usi.si.seart.src2abs;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.MethodReferenceExpr;
import com.github.javaparser.ast.type.Type;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.experimental.FieldDefaults;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

@Getter
@RequiredArgsConstructor
@FieldDefaults(level = AccessLevel.PRIVATE, makeFinal = true)
class Parser {

	public enum Granularity {
		METHOD, CLASS
	}

	Set<String> types = new HashSet<>();
	Set<String> methods = new HashSet<>();
	Set<String> annotations = new HashSet<>();

	Granularity granularity;

	public void parse(String sourceCode) {
		Function<String, Node> parsingFunction;
		switch (granularity) {
			case CLASS:
				parsingFunction = StaticJavaParser::parse;
				break;
			case METHOD:
				parsingFunction = StaticJavaParser::parseMethodDeclaration;
				break;
			default:
				throw new UnsupportedOperationException("Parsing not supported at '" + granularity + "'");
		}

		parse(sourceCode, parsingFunction);
	}

	public void parse(String sourceCode, Function<String, Node> parsingFunction) {
		Node node = parsingFunction.apply(sourceCode);
		traverseNode(node);
	}

	private void traverseNode(Node node) {
		// create set of annotations
		node.findAll(AnnotationExpr.class).stream()
				.map(AnnotationExpr::getNameAsString)
				.forEach(annotations::add);

		// create set of types
		node.findAll(Type.class).stream()
				.flatMap(type -> {
					String[] allTypes = filterString(type.asString());
					return Stream.of(allTypes);
				})
				.forEach(types::add);

		// create set of methods and insert declared methods
		node.findAll(MethodDeclaration.class).stream()
				.map(MethodDeclaration::getNameAsString)
				.forEach(methods::add);

		// insert referenced methods into methods set
		node.findAll(MethodCallExpr.class).stream()
				.map(MethodCallExpr::getNameAsString)
				.forEach(methods::add);

		// insert scope of methods into types
		node.findAll(MethodCallExpr.class).stream()
				.map(MethodCallExpr::getScope)
				.flatMap(Optional::stream)
				.filter(expression -> expression.isFieldAccessExpr() || expression.isNameExpr())
				.forEach(expression -> {
					String exprStr = expression.toString();
					String[] fragments = exprStr.split("\\.");
					String[] letters = fragments[fragments.length - 1].split("");
					if (!letters[0].equals(letters[0].toLowerCase())) {
						String[] allTypes = filterString(exprStr);
						types.addAll(Arrays.asList(allTypes));
					}
				});

		// insert scope of methods into types
		node.findAll(MethodReferenceExpr.class).stream()
				.map(MethodReferenceExpr::getIdentifier)
				.forEach(methods::add);
	}

	private String[] filterString(String typeString){
		String[] listString;
		typeString = typeString.replaceAll("\\[", " ");
		typeString = typeString.replaceAll("\\]", " ");
		typeString = typeString.replaceAll("\\(", " ");
		typeString = typeString.replaceAll("\\)", " ");
		typeString = typeString.replaceAll(">", " ");
		typeString = typeString.replaceAll("<", " ");
		typeString = typeString.replaceAll("\\{", " ");
		typeString = typeString.replaceAll("\\}", " ");
		typeString = typeString.replaceAll("\\,", " ");
		typeString = typeString.replaceAll("\\?", " ");
		typeString = typeString.replaceAll("extends", " ");
		typeString = typeString.replaceAll("implements", " ");
		listString = typeString.split(" ");
		return listString;
	}
}
