package ch.viseon.openOrca.client.impl

import ch.viseon.openOrca.client.just
import ch.viseon.openOrca.share.*
import ch.viseon.openOrca.share.impl.DefaultPresentationModel
import kodando.rxjs.Rx
import kodando.rxjs.flatMap

interface CommandListExecutor {

  fun execute(modelStore: DefaultPresentationModelStore, commandData: Rx.IObservable<CommandData>): Rx.IObservable<Event>

  fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    return execute(modelStore, just(commandData))
  }

}

class ClientCommandListExecutor : CommandListExecutor {

  private val executors = mapOf(
      CreateModelCommandData::class.simpleName!! to CreateModelExecutor,
      RemoveModelCommandData::class.simpleName!! to RemoveModelExecutor,
      ActionCommandData::class.simpleName!! to ActionExecutor,
      ChangeValueCommandData::class.simpleName!! to ChangeValueExecutor
  )

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: Rx.IObservable<CommandData>): Rx.IObservable<Event> {
    return commandData.flatMap { applyCommand(modelStore, it) }
  }

  private fun applyCommand(modelStore: DefaultPresentationModelStore, it: CommandData): Rx.IObservable<Event> {
    val executor = executors[it::class.simpleName!!] ?: throw IllegalArgumentException("No executor found for commandData: '$it'")
    return executor.execute(modelStore, it)
  }
}

interface CommandExecutor {
  fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event>
}

private object CreateModelExecutor : CommandExecutor {

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as CreateModelCommandData

    val model = DefaultPresentationModel(it.modelId, it.modelType, it.properties.asSequence())
    modelStore.addModel(model)
    return just(ModelStoreChangeEvent(it.source, model.id, model.type, ModelStoreChangeEvent.Type.ADD))
  }
}

object ChangeValueExecutor: CommandExecutor {
  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as ChangeValueCommandData
    val model = modelStore[it.modelId]
    val oldValue = model[it.propertyName].setValue(it.value)
    return just(PropertyChangeEvent(it.source, model.id, model.type, ValueChangeEvent(it.source, it.propertyName, oldValue, it.value)))
  }
}

object ActionExecutor: CommandExecutor {

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as ActionCommandData
    return just(ActionEvent(it.source, it.actionName, it.modelIds))
  }
}

object RemoveModelExecutor: CommandExecutor {
  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as RemoveModelCommandData
    val oldModel = modelStore.removeModel(it.modelId)
    return just(ModelStoreChangeEvent(it.source, it.modelId, oldModel.type, ModelStoreChangeEvent.Type.REMOVE))
  }

}
