/*
 * Copyright 2017 viseon gmbh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ch.viseon.openOrca.client

import ch.viseon.openOrca.share.*
import ch.viseon.openOrca.share.impl.DefaultPresentationModel
import ch.viseon.openOrca.share.impl.DefaultPresentationModelStore
import kodando.rxjs.Rx
import kodando.rxjs.flatMap

interface CommandListExecutor {

  fun execute(modelStore: DefaultPresentationModelStore, commandData: Rx.IObservable<CommandData>): Rx.IObservable<Event>

  fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    return execute(modelStore, just(commandData))
  }

}

class ClientCommandListExecutor : CommandListExecutor {

  private val executors = mapOf(
      CreateModelCommandData::class.simpleName!! to CreateModelExecutor,
      RemoveModelCommandData::class.simpleName!! to RemoveModelExecutor,
      ActionCommandData::class.simpleName!! to ActionExecutor,
      ChangeValueCommandData::class.simpleName!! to ChangeValueExecutor,
      RemoveModelByTypeCommandData::class.simpleName!! to RemoveModelByTypeExecutor,
      SyncModelCommandData::class.simpleName!! to SyncModelExecutor
  )

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: Rx.IObservable<CommandData>): Rx.IObservable<Event> {
    return commandData.flatMap { applyCommand(modelStore, it) }
  }

  private fun applyCommand(modelStore: DefaultPresentationModelStore, it: CommandData): Rx.IObservable<Event> {
    val executor = executors[it::class.simpleName!!] ?: throw IllegalArgumentException("No executor found for commandData: '$it'")
    return executor.execute(modelStore, it)
  }
}

interface CommandExecutor {
  fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event>
}

private object CreateModelExecutor : CommandExecutor {

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as CreateModelCommandData

    val model = DefaultPresentationModel(it.modelId, it.modelType, it.properties.asSequence())
    modelStore.addModel(model)
    return just(ModelStoreChangeEvent(it.source, model.id, model.type, ModelStoreChangeEventType.ADD))
  }
}

private object ChangeValueExecutor : CommandExecutor {
  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as ChangeValueCommandData
    val model = modelStore[it.modelId]
    val oldValue = model[it.propertyName].setValue(it.value)
    return just(PropertyChangeEvent(it.source, model.id, model.type, ValueChangeEvent(it.source, it.propertyName, oldValue, it.value)))
  }
}

private object ActionExecutor : CommandExecutor {

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as ActionCommandData
    return just(ActionEvent(it.source, it.actionName, it.modelIds))
  }
}

private object RemoveModelExecutor : CommandExecutor {
  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val it = commandData as RemoveModelCommandData
    val oldModel = modelStore.removeModel(it.modelId)
    return just(ModelStoreChangeEvent(it.source, it.modelId, oldModel.type, ModelStoreChangeEventType.REMOVE))
  }

}

private object RemoveModelByTypeExecutor : CommandExecutor {

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val data = commandData as RemoveModelByTypeCommandData
    val map: Iterable<Event> = modelStore.removeModels(data.modelType).map { ModelStoreChangeEvent(data.source, it.id, it.type, ModelStoreChangeEventType.REMOVE) }
    return Rx.Observable.from(map.toList().toTypedArray())
  }

}

private object SyncModelExecutor : CommandExecutor {

  override fun execute(modelStore: DefaultPresentationModelStore, commandData: CommandData): Rx.IObservable<Event> {
    val data = commandData as SyncModelCommandData
    val sourcePm = modelStore[data.sourceModel]
    val destinationPm = modelStore[data.destinationModel]
    val events: List<Event> = sourcePm.getProperties()
        .flatMap { property: Property ->
          property.getValuesWithTypes()
              .map { it: Pair<String, PropertyValue> -> it.second }
              .map { value: PropertyValue ->
                //Values are immutable
                val oldValue = destinationPm[property.name].setValue(value)
                ValueChangeEvent(data.source, property.name, oldValue, value)
              }
        }
        .toList()
    return Rx.Observable.from(events.toTypedArray())
  }

}
