/*
 * Copyright 2017 viseon gmbh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ch.viseon.openOrca.client

import ch.viseon.openOrca.share.*
import kodando.rxjs.Rx
import kodando.rxjs.filter
import kodando.rxjs.map

class ClientOrcaImpl(val commandListExecutor: CommandListExecutor, val transmitter: ch.viseon.openOrca.client.Transmitter) : ClientOrca {

  val clientModelStore = ch.viseon.openOrca.share.impl.DefaultPresentationModelStore()

  private val eventSubject = kodando.rxjs.Rx.Subject<Event>()

  override fun executeCommands(commands: Array<ch.viseon.openOrca.share.CommandData>) {
    val responseCommands = transmitter.sendCommands(commands)
    val responseEvent: kodando.rxjs.Rx.IObservable<Event> = commandListExecutor.execute(
        clientModelStore,
        kodando.rxjs.Rx.Observable.Companion.merge(Rx.Observable.from(commands), responseCommands)
    )

    responseEvent.subscribe {
      eventSubject.next(it)
    }
  }

  override fun observeModelStore(): kodando.rxjs.Rx.IObservable<ModelStoreChangeEvent> {
    return eventSubject
        .filter { it is ch.viseon.openOrca.share.ModelStoreChangeEvent }
        .map { it as ch.viseon.openOrca.share.ModelStoreChangeEvent }
  }

  override fun observeModel(modelId: ch.viseon.openOrca.share.ModelId): kodando.rxjs.Rx.IObservable<PropertyChangeEvent> {
    return eventStreamAsPropertyChangeEvent()
        .filter { it.modelId == modelId }
  }

  override fun observeModel(modelType: ch.viseon.openOrca.share.ModelType): kodando.rxjs.Rx.IObservable<PropertyChangeEvent> {
    return eventStreamAsPropertyChangeEvent()
        .filter { it.modelType == modelType }
  }

  override fun observeProperty(modelId: ch.viseon.openOrca.share.ModelId, propertyName: ch.viseon.openOrca.share.PropertyName): kodando.rxjs.Rx.IObservable<ValueChangeEvent> {
    return eventStreamAsPropertyChangeEvent()
        .filter { it.modelId == modelId }
        .map { it.valueChangeEvent }
  }

  private fun eventStreamAsPropertyChangeEvent(): kodando.rxjs.Rx.IObservable<PropertyChangeEvent> {
    return eventSubject
        .filter { it is ch.viseon.openOrca.share.PropertyChangeEvent }
        .map { it as ch.viseon.openOrca.share.PropertyChangeEvent }
  }

  override fun registerNamedCommand(actionName: String): kodando.rxjs.Rx.IObservable<ActionEvent> {
    return eventSubject
        .filter { it is ch.viseon.openOrca.share.ActionEvent }
        .map { it as ch.viseon.openOrca.share.ActionEvent }
        .filter { it.actionName == actionName }
  }

  override fun model(modelType: ch.viseon.openOrca.share.ModelType): Array<ch.viseon.openOrca.share.PresentationModel> {
    return clientModelStore[modelType].toTypedArray()
  }

  override fun model(modelId: ch.viseon.openOrca.share.ModelId): ch.viseon.openOrca.share.PresentationModel {
    return clientModelStore[modelId]
  }
}

