package ch.viseon.openOrca.server.impl

import ch.viseon.openOrca.share.*
import ch.viseon.openOrca.share.impl.DefaultPresentationModel
import io.reactivex.Observable

interface CommandListExecutor {

  fun execute(modelStore: ModelStore, commandData: Observable<CommandData>): Observable<Event>

  fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    return execute(modelStore, Observable.just(commandData))
  }

}

class ServerCommandListExecutor : CommandListExecutor {

  private val executors = mapOf(
      CreateModelCommandData::class.simpleName!! to CreateModelExecutor,
      RemoveModelCommandData::class.simpleName!! to RemoveModelExecutor,
      ActionCommandData::class.simpleName!! to ActionExecutor,
      ChangeValueCommandData::class.simpleName!! to ChangeValueExecutor
  )

  override fun execute(modelStore: ModelStore, commandData: Observable<CommandData>): Observable<Event> {
    return commandData.flatMap { applyCommand(modelStore, it) }
  }

  private fun applyCommand(modelStore: ModelStore, it: CommandData): Observable<Event> {
    val executor = executors[it::class.simpleName!!] ?: throw IllegalArgumentException("No executor found for commandData: '$it'")
    return executor.execute(modelStore, it)
  }

}

interface CommandExecutor {
  fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event>
}

private object CreateModelExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    val it = commandData as CreateModelCommandData

    val model = DefaultPresentationModel(it.modelId, it.modelType, it.properties.asSequence())
    modelStore.addModel(model)
    return Observable.just(ModelStoreChangeEvent(it.source, model.id, model.type, ModelStoreChangeEvent.Type.ADD))
  }
}

object ChangeValueExecutor: CommandExecutor {
  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    val it = commandData as ChangeValueCommandData
    val model = modelStore[it.modelId]
    val oldValue = model[it.propertyName].setValue(it.value)
    return Observable.just(PropertyChangeEvent(it.source, model.id, model.type, ValueChangeEvent(it.source, it.propertyName, oldValue, it.value)))
  }
}

object ActionExecutor: CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    val it = commandData as ActionCommandData
    return Observable.just(ActionEvent(it.source, it.actionName, it.modelIds))
  }
}

object RemoveModelExecutor: CommandExecutor {
  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    val it = commandData as RemoveModelCommandData
    val oldModel = modelStore.removeModel(it.modelId)
    return Observable.just(ModelStoreChangeEvent(it.source, it.modelId, oldModel.type, ModelStoreChangeEvent.Type.REMOVE))
  }

}
