/*
 * Copyright 2017 viseon gmbh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ch.viseon.openOrca.server.impl

import ch.viseon.openOrca.share.*
import ch.viseon.openOrca.share.impl.DefaultPresentationModel
import ch.viseon.openOrca.share.impl.Property
import io.reactivex.Observable

interface CommandListExecutor {

  fun execute(modelStore: ModelStore, commandData: Observable<CommandData>): Observable<Event>

  fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    return execute(modelStore, Observable.just(commandData))
  }

}

private val LOGGER = loggerFor<ServerCommandListExecutor>()

class ServerCommandListExecutor : CommandListExecutor {

  private val executors = mapOf(
      CreateModelCommandData::class.simpleName!! to CreateModelExecutor,
      RemoveModelCommandData::class.simpleName!! to RemoveModelExecutor,
      ActionCommandData::class.simpleName!! to ActionExecutor,
      ChangeValueCommandData::class.simpleName!! to ChangeValueExecutor,
      RemoveModelByTypeCommandData::class.simpleName!! to RemoveModelByTypeExecutor,
      SyncModelCommandData::class.simpleName!! to SyncModelExecutor
  )

  override fun execute(modelStore: ModelStore, commandData: Observable<CommandData>): Observable<Event> {
    return commandData.flatMap { applyCommand(modelStore, it) }
  }

  private fun applyCommand(modelStore: ModelStore, it: CommandData): Observable<Event> {
    LOGGER.finer { "applyingCommand data: $it" }

    val executor = executors[it::class.simpleName!!] ?: throw IllegalArgumentException("No executor found for commandData: '$it'")
    return executor.execute(modelStore, it)
  }

}

interface CommandExecutor {
  fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event>
}

private object CreateModelExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    LOGGER.finer { "applyingCommand data: $commandData" }
    val it = commandData as CreateModelCommandData

    val model = DefaultPresentationModel(it.modelId, it.modelType, it.properties.asSequence())
    modelStore.addModel(model)
    return Observable.just(ModelStoreChangeEvent(it.source, model.id, model.type, ModelStoreChangeEvent.Type.ADD))
  }
}

private object ChangeValueExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    LOGGER.finer { "Execute ChangeValue. Data: '$commandData" }

    val it = commandData as ChangeValueCommandData
    val model = modelStore[it.modelId]
    val oldValue = model[it.propertyName].setValue(it.value)
    return Observable.just(PropertyChangeEvent(it.source, model.id, model.type, ValueChangeEvent(it.source, it.propertyName, oldValue, it.value)))
  }

}

private object ActionExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    LOGGER.finer { "Execute Action. Data: '$commandData" }

    val it = commandData as ActionCommandData
    return Observable.just(ActionEvent(it.source, it.actionName, it.modelIds))
  }

}

private object RemoveModelExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    LOGGER.finer { "Execute RemoveModel. Data: '$commandData" }
    val it = commandData as RemoveModelCommandData
    val oldModel = modelStore.removeModel(it.modelId)
    return Observable.just(ModelStoreChangeEvent(it.source, it.modelId, oldModel.type, ModelStoreChangeEvent.Type.REMOVE))
  }

}

private object RemoveModelByTypeExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    LOGGER.finer { "Execute RemoveModelByType. Data: '$commandData" }
    val data = commandData as RemoveModelByTypeCommandData
    return Observable.fromIterable(modelStore.removeModels(data.modelType).map { ModelStoreChangeEvent(data.source, it.id, it.type, ModelStoreChangeEvent.Type.REMOVE) })
  }

}

private object SyncModelExecutor : CommandExecutor {

  override fun execute(modelStore: ModelStore, commandData: CommandData): Observable<Event> {
    LOGGER.finer { "Execute SyncModel. Data: '$commandData" }

    val data = commandData as SyncModelCommandData
    val sourcePm = modelStore[data.sourceModel]
    val destinationPm = modelStore[data.destinationModel]
    val events = sourcePm.getProperties()
        .flatMap { property: Property ->
          property.getValues()
              .map { it: Map.Entry<String, PropertyValue> -> it.value }
              .map { value: PropertyValue ->
                //Values are immutable
                val oldValue = destinationPm[property.name].setValue(value)
                ValueChangeEvent(data.source, property.name, oldValue, value)
              }
        }
        .toList()
    return Observable.fromIterable(events)
  }

}
