/*
 * Copyright 2017 viseon gmbh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ch.viseon.openOrca.server.servlet

import ch.viseon.openOrca.server.*
import ch.viseon.openOrca.server.impl.ServerCommandListExecutor
import ch.viseon.openOrca.share.CommandData
import ch.viseon.openOrca.share.impl.DefaultPresentationModelStore
import com.google.gson.Gson
import com.google.gson.GsonBuilder
import io.reactivex.Observable
import javax.servlet.http.HttpServlet
import javax.servlet.http.HttpServletRequest
import javax.servlet.http.HttpServletResponse

abstract class DefaultServlet : HttpServlet() {

  override fun doPost(req: HttpServletRequest, resp: HttpServletResponse) {
    val run = OrcaRun(ServerCommandListExecutor(), this::processRequest, ServletStoreLifeCycle(req))
    run.execute(ServletRequest(req), ServletResponse(resp))
  }

  protected abstract fun processRequest(orcaSource: OrcaSource): Observable<CommandData>

}

private class ServletResponse(private val resp: HttpServletResponse) : Response {
  override fun sendCommands(commands: Observable<CommandData>) {
    val toSend = mutableListOf<CommandData>()
    commands.subscribe {
      toSend.add(it)
    }
    val toSendArray = toSend.toTypedArray()
    val gson = createGson()
    val json = gson.toJson(toSendArray)
    resp.status = 200
    resp.writer.use { println(json) }
  }

}

private fun createGson(): Gson {
  return GsonBuilder().create()
}

private class ServletRequest(private val req: HttpServletRequest) : Request {

  override fun readCommands(): Observable<CommandData> {
    val commands = req.reader.use {
      createGson().fromJson<Array<CommandData>>(it, Array<CommandData>::class.java)
    }
    return Observable.fromArray(*commands)
  }

}

class ServletStoreLifeCycle(private val req: HttpServletRequest) : StoreLifecycle {

  companion object {
    @JvmStatic
    val MODEL_STORE = "modelStore"
  }

  override fun getStore(): DefaultPresentationModelStore {
    return req.getSession(false)?.let {
      it.getAttribute(MODEL_STORE) as DefaultPresentationModelStore
    } ?: DefaultPresentationModelStore().also {
      req.session.setAttribute(MODEL_STORE, it)
    }
  }

  override fun updateStore(modelStore: DefaultPresentationModelStore) {
    req.session.setAttribute(MODEL_STORE, modelStore)
  }

}
