/*
 * Copyright 2017 viseon gmbh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ch.viseon.openOrca.share

class Property(val name: ch.viseon.openOrca.share.PropertyName, initialValues: Sequence<PropertyValue>) {

  private val propertyValue2Values: MutableMap<String, PropertyValue> = HashMap<String, PropertyValue>().apply {
    initialValues.forEach { value -> this[value::class.simpleName!!] = value }
  }

  /**
   * @return the oldValue of the given propertyValue.
   */
  fun setValue(value: PropertyValue): PropertyValue? {
    val typeName = getTypeName(value::class.simpleName!!)
    //println("setter: typeName: '$typeName' value: '$value'")
    val oldValue: PropertyValue? = propertyValue2Values[typeName]
    propertyValue2Values[typeName] = value
    return oldValue
  }

  fun getValuesWithTypes(): Array<Pair<String, PropertyValue>> {
    return propertyValue2Values.entries.asSequence().map { Pair(it.key, it.value) }.toList().toTypedArray()
  }

  fun getValues(): Array<PropertyValue> {
    return propertyValue2Values.values.toList().toTypedArray()
  }

  operator fun get(valueType: String): PropertyValue? {
    val typeName = getTypeName(valueType)
    //println("getter: typeName: '$typeName' value: '$valueType'")
    return propertyValue2Values[typeName]
  }

  inline fun <reified T : PropertyValue> getValue(): T {
    val get = get(T::class.simpleName!!) ?: throw IllegalStateException("Value is null for type: ${T::class}")
    return get as? T ?: throw IllegalArgumentException("Value is not of type ${T::class}. Actual value '$get' Type: ${get.let { it::class.simpleName }}")
  }

  private fun getTypeName(valueType: String): String {
    val typeName = if (valueType == ch.viseon.openOrca.share.StringValue::class.simpleName
        || valueType == ch.viseon.openOrca.share.BooleanValue::class.simpleName
        || valueType == ch.viseon.openOrca.share.ModelIdValue::class.simpleName
                       ) {
      ch.viseon.openOrca.share.ValuePropertyValue::class.simpleName!!
    } else {
      valueType
    }
    return typeName
  }

  inline fun <reified T> contains(): Boolean {
    return get(T::class.simpleName!!) != null
  }

  override fun toString(): String {
    return "$name: values: $propertyValue2Values"
  }

}
