/*
 * Copyright 2017 viseon gmbh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ch.viseon.openOrca.share

class JsonArray<T>(val values: Iterable<T>) : Iterable<T> by values

interface JsonObj {

  operator fun set(key: String, value: String)
  operator fun set(key: String, value: Boolean)
  operator fun set(key: String, block: JsonObj.() -> Unit)
  operator fun set(key: String, value: JsonObj)
  operator fun set(key: String, value: JsonArray<JsonObj>)
  operator fun set(key: String, value: Iterable<String>)

  fun string(key: String): String
  fun boolean(key: String): Boolean

  fun getObject(key: String): JsonObj

  fun <T> pArray(key: String): JsonArray<T>
  fun oArray(key: String): JsonArray<JsonObj>

  fun toJsonString(): String
  fun integer(key: String): Int
  fun long(key: String): Long
  fun float(key: String): Float
  fun double(key: String): Double

}

interface JsonFactory {
  fun parse(content: String): JsonArray<JsonObj>

  fun newJson(block: JsonObj.() -> Unit): JsonObj
}

class JsonCodec(private val jsonFactory: JsonFactory) : StringCodec {

  override val mimeType = "application/json"
  override val encoding = "UTF-8"

  private val commandCodecs = mapOf(
      ActionCommandData::class.simpleName!! to ActionCommandCodec(jsonFactory),
      RemoveModelCommandData::class.simpleName!! to RemoveModelCommandCodec(jsonFactory),
      RemoveModelByTypeCommandData::class.simpleName!! to RemoveModelByTypeCodec(jsonFactory),
      ChangeValueCommandData::class.simpleName!! to ChangeValueCommandCodec(jsonFactory),
      CreateModelCommandData::class.simpleName!! to CreateModelCommandCodec(jsonFactory),
      SyncModelCommandData::class.simpleName!! to SyncModelCommandCodec(jsonFactory)

  )

  override fun decode(source: Source, data: String): Iterable<CommandData> {
    val json = jsonFactory.parse(data)
    val result = json.map {
      val type = it.string("type")
      commandCodecs[type]!!.decode(source, it)
    }
    return result
  }

  override fun encode(commands: Iterable<CommandData>): String {
    return buildString {
      append("[")
      append(
          commands
              .asSequence()
              .map { commandCodecs[it::class.simpleName!!]!!.encode(it) }
              .map { it.toJsonString() }
              .joinToString(",")
      )
      append("]")
    }
  }

}


interface CommandCodec {
  fun encode(untypedCommandData: CommandData): JsonObj
  fun decode(source: Source, json: JsonObj): CommandData
}

private class ActionCommandCodec(val jsonFactory: JsonFactory) : CommandCodec {

  override fun encode(untypedCommandData: CommandData): JsonObj {
    val typedCommand = untypedCommandData as ActionCommandData

    val jsonObj = jsonFactory.newJson {
      this["type"] = ActionCommandData::class.simpleName!!
      this["data"] = {
        this["actionName"] = typedCommand.actionName
        this["modelIds"] = typedCommand.modelIds.map { it.stringId }
      }
    }

    return jsonObj
  }

  override fun decode(source: Source, json: JsonObj): CommandData {
    val data = json.getObject("data")
    val modelIds: JsonArray<String> = data.pArray("modelIds")
    return ActionCommandData(source, data.string("actionName"), modelIds.map(::ModelId))
  }

}

private class RemoveModelCommandCodec(private val jsonFactory: JsonFactory) : CommandCodec {

  override fun encode(untypedCommandData: CommandData): JsonObj {
    val typedCommand = untypedCommandData as RemoveModelCommandData

    return jsonFactory.newJson {
      this["type"] = RemoveModelCommandData::class.simpleName!!
      this["data"] = {
        this["modelId"] = typedCommand.modelId.stringId
      }
    }
  }

  override fun decode(source: Source, json: JsonObj): CommandData {
    val data = json.getObject("data")
    return RemoveModelCommandData(source, ModelId(data.string("modelId")))
  }

}

private class ChangeValueCommandCodec(private val jsonFactory: JsonFactory) : CommandCodec {

  override fun encode(untypedCommandData: CommandData): JsonObj {
    val changeValueCommand = untypedCommandData as ChangeValueCommandData

    return jsonFactory.newJson {
      this["type"] = ChangeValueCommandData::class.simpleName!!
      this["data"] = {
        this["modelId"] = changeValueCommand.modelId.stringId
        this["propertyName"] = changeValueCommand.propertyName.name
        this["value"] = jsonFactory.newJson { valueToJson(this, changeValueCommand.tag, changeValueCommand.value) }
      }
    }
  }

  override fun decode(source: Source, json: JsonObj): CommandData {
    val data: JsonObj = json.getObject("data")
    val modelId = ModelId(data.string("modelId"))
    val propertyName = PropertyName(data.string("propertyName"))
    val (tag, value) = valueFromJson(data.getObject("value"))
    return ChangeValueCommandData(source, modelId, propertyName, tag, value)
  }

}

private class CreateModelCommandCodec(private val jsonFactory: JsonFactory) : CommandCodec {

  override fun encode(untypedCommandData: CommandData): JsonObj {
    val createCommand = untypedCommandData as CreateModelCommandData

    return jsonFactory.newJson {
      this["type"] = CreateModelCommandData::class.simpleName!!
      this["data"] = {
        this["modelId"] = createCommand.modelId.stringId
        this["type"] = createCommand.modelType.stringId
        this["properties"] = JsonArray(
            createCommand.properties
                .map {
                  jsonFactory.newJson {
                    this["propertyName"] = it.name.name
                    this["values"] = JsonArray(it.getValues()
                        .map { (tag, value) -> jsonFactory.newJson { valueToJson(this, tag, value) } }
                        .toList())
                  }
                })
      }
    }
  }

  override fun decode(source: Source, json: JsonObj): CommandData {
    val data = json.getObject("data")
    val properties: JsonArray<JsonObj> = data.oArray("properties")
    return CreateModelCommandData(
        source,
        ModelId(data.string("modelId")),
        ModelType(data.string("type")),
        properties.map {
          val values: JsonArray<JsonObj> = it.oArray("values")
          Property(
              PropertyName(it.string("propertyName")),
              values.asSequence().map(::valueFromJson))
        })
  }

}

class RemoveModelByTypeCodec(private val jsonFactory: JsonFactory) : CommandCodec {

  override fun encode(untypedCommandData: CommandData): JsonObj {
    val commandData = untypedCommandData as RemoveModelByTypeCommandData

    return jsonFactory.newJson {
      this["type"] = RemoveModelByTypeCommandData::class.simpleName!!
      this["data"] = {
        this["modelTypeId"] = commandData.modelType.stringId
      }
    }
  }

  override fun decode(source: Source, json: JsonObj): CommandData {
    val data = json.getObject("data")
    return RemoveModelByTypeCommandData(source, ModelType(data.string("modelTypeId")))
  }

}

class SyncModelCommandCodec(private val jsonFactory: JsonFactory) : CommandCodec {

  override fun encode(untypedCommandData: CommandData): JsonObj {
    val commandData = untypedCommandData as SyncModelCommandData

    return jsonFactory.newJson {
      this["type"] = SyncModelCommandData::class.simpleName!!
      this["data"] = {
        this["src"] = commandData.sourceModel.stringId
        this["dst"] = commandData.destinationModel.stringId
      }
    }
  }

  override fun decode(source: Source, json: JsonObj): CommandData {
    val data = json.getObject("data")
    return SyncModelCommandData(source, ModelId(data.string("src")), ModelId(data.string("dst")))
  }

}

fun valueToJson(jsonObj: JsonObj, tag: Tag, value: PropertyValue): JsonObj {
  with(jsonObj) {
    this["type"] = value.getJsonTypeId()
    this["value"] = value.asJsonValue()
    this["tag"] = tag.name
  }
  return jsonObj
}

fun valueFromJson(json: JsonObj): Pair<Tag, PropertyValue> {
  val type = json.string("type")
  val value = when (type) {
    StringValue::class.simpleName!! -> StringValue(json.string("value"))
    BooleanValue::class.simpleName!! -> BooleanValue(json.boolean("value"))
    ModelIdValue::class.simpleName!! -> ModelIdValue(ModelId(json.string("value")))
    IntegerValue::class.simpleName!! -> IntegerValue(json.integer("value"))
    FloatValue::class.simpleName!! -> FloatValue(json.float("value"))
    DoubleValue::class.simpleName!! -> DoubleValue(json.double("value"))
    else -> {
      throw IllegalArgumentException("Unknown propertyValue type: '$type'")
    }
  }

  val tag = json.string("tag")
  return Pair(Tag(tag), value)
}
