package cn.allbs.captcha.core;

import cn.allbs.captcha.enums.ImageEnum;
import cn.allbs.captcha.generator.CaptchaGenerator;
import cn.allbs.captcha.utils.InterferingUtil;
import cn.hutool.core.codec.Base64;
import cn.hutool.core.img.ImgUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.IORuntimeException;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.URLUtil;

import java.awt.*;
import java.awt.geom.CubicCurve2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;

/**
 * @author ChenQi
 */
public abstract class AbstractCaptcha implements ICaptcha {

    /**
     * 图片的宽度
     */
    protected int width;
    /**
     * 图片的高度
     */
    protected int height;
    /**
     * 验证码干扰元素个数
     */
    protected int interfereCount;
    /**
     * 字体
     */
    protected Font font;
    /**
     * 验证码
     */
    protected String code;
    /**
     * 验证码图片
     */
    protected byte[] imageBytes;
    /**
     * 验证码生成器
     */
    protected CaptchaGenerator generator;
    /**
     * 背景色
     */
    protected Color background;
    /**
     * 文字透明度
     */
    protected AlphaComposite textAlpha;
    /**
     * 字符个数
     */
    protected int len;
    /**
     * 文字颜色
     */
    protected Color color;
    /**
     * code
     */
    protected char[] codeChar;
    /**
     * 图片类型
     */
    protected int imageType;
    /**
     * contentType
     */
    protected String mimeType;

    public AbstractCaptcha(int width, int height, CaptchaGenerator generator, int interfereCount, int len) {
        this.width = width;
        this.height = height;
        this.generator = generator;
        this.interfereCount = interfereCount;
        this.len = len;
        // 字体高度设为验证码高度-2，留边距
        this.font = new Font(Font.SANS_SERIF, Font.PLAIN, (int) (this.height * 0.75));
    }

    public AbstractCaptcha(int width, int height, CaptchaGenerator generator, int interfereCount, int len, Font font) {
        this.width = width;
        this.height = height;
        this.generator = generator;
        this.interfereCount = interfereCount;
        this.len = len;
        this.font = font;
    }

    public AbstractCaptcha(CaptchaDetails captchaDetails) {
        this.width = captchaDetails.width();
        this.height = captchaDetails.height();
        this.len = captchaDetails.len();
        this.font = captchaDetails.font();
        this.interfereCount = captchaDetails.interfereCount();
    }

    public AbstractCaptcha(CaptchaDetails captchaDetails, CaptchaGenerator generator) {
        this.width = captchaDetails.width();
        this.height = captchaDetails.height();
        this.len = captchaDetails.len();
        this.font = captchaDetails.font();
        this.interfereCount = captchaDetails.interfereCount();
        this.imageType = captchaDetails.imageType();
        this.background = captchaDetails.color();
        this.generator = generator;
    }

    @Override
    public void create() {
        generateCode();

        if (imageType == ImageEnum.JPG.getType()) {
            this.mimeType = "image/jpg";
            final ByteArrayOutputStream out = new ByteArrayOutputStream();
            ImgUtil.writeJpg(createImage(this.codeChar), out);
            this.imageBytes = out.toByteArray();
        }
        if (imageType == ImageEnum.PNG.getType()) {
            this.mimeType = "image/png";
            final ByteArrayOutputStream out = new ByteArrayOutputStream();
            ImgUtil.writePng(createImage(this.codeChar), out);
            this.imageBytes = out.toByteArray();
        }
        if (imageType == ImageEnum.GIF.getType()) {
            this.mimeType = "image/gif";
            createImageGif(this.codeChar);
        }

    }

    @Override
    public String getText() {
        if (null == this.code) {
            create();
        }
        return this.code;
    }

    @Override
    public boolean verify(String checkedCode) {
        return this.generator.verify(getText(), checkedCode);
    }

    @Override
    public void write(OutputStream out) {
        IoUtil.write(out, false, getImageBytes());
    }

    /**
     * 验证码写出到文件
     *
     * @param path 文件路径
     * @throws IORuntimeException IO异常
     */
    public void write(String path) throws IORuntimeException {
        this.write(FileUtil.touch(path));
    }

    /**
     * 验证码写出到文件
     *
     * @param file 文件
     * @throws IORuntimeException IO异常
     */
    public void write(File file) throws IORuntimeException {
        try (OutputStream out = FileUtil.getOutputStream(file)) {
            this.write(out);
        } catch (IOException e) {
            throw new IORuntimeException(e);
        }
    }

    /**
     * 生成验证码
     *
     * @since 3.3.0
     */
    protected void generateCode() {
        this.codeChar = generator.generator();
        this.code = String.valueOf(this.codeChar);
    }

    /**
     * 根据生成的code创建验证码图片
     *
     * @param code 验证码
     * @return Image
     */
    protected abstract Image createImage(char[] code);

    /**
     * 创建验证码gif图
     *
     * @param code 验证码
     */
    protected abstract void createImageGif(char[] code);

    /**
     * 获取图形验证码图片bytes
     *
     * @return 图形验证码图片bytes
     * @since 4.5.17
     */
    public byte[] getImageBytes() {
        if (null == this.imageBytes) {
            create();
        }
        return this.imageBytes;
    }

    /**
     * 获取验证码图
     *
     * @return 验证码图
     */
    public BufferedImage getImage() {
        return ImgUtil.read(IoUtil.toStream(getImageBytes()));
    }

    /**
     * 获得图片的Base64形式
     *
     * @return 图片的Base64
     * @since 3.3.0
     */
    public String getImageBase64() {
        return Base64.encode(getImageBytes());
    }

    /**
     * 获取图片带文件格式的 Base64
     *
     * @return 图片带文件格式的 Base64
     * @since 5.3.11
     */
    public String getImageBase64Data() {
        return URLUtil.getDataUriBase64(this.mimeType, getImageBase64());
    }

    /**
     * 获取图片mimeType
     *
     * @return 图片mimeType
     */
    public String getMimeType() {
        return this.mimeType;
    }

    /**
     * 自定义字体
     *
     * @param font 字体
     */
    public void setFont(Font font) {
        this.font = font;
    }

    /**
     * 获取验证码生成器
     *
     * @return 验证码生成器
     */
    public CaptchaGenerator getGenerator() {
        return generator;
    }

    /**
     * 设置验证码生成器
     *
     * @param generator 验证码生成器
     */
    public void setGenerator(CaptchaGenerator generator) {
        this.generator = generator;
    }

    /**
     * 设置背景色
     *
     * @param background 背景色
     * @since 4.1.22
     */
    public void setBackground(Color background) {
        this.background = background;
    }

    /**
     * 设置文字透明度
     *
     * @param textAlpha 文字透明度，取值0~1，1表示不透明
     * @since 4.5.17
     */
    public void setTextAlpha(float textAlpha) {
        this.textAlpha = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, textAlpha);
    }

    /**
     * 画随机码图
     *
     * @param fontColor 随机字体颜色
     * @param strs      字符数组
     * @param flag      透明度
     * @param besselXY  干扰线参数
     * @return BufferedImage
     */
    public BufferedImage graphicsImage(Color[] fontColor, char[] strs, int flag, int[][] besselXY) {
        BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = (Graphics2D) image.getGraphics();
        // 填充背景颜色
        g2d.setColor(background);
        g2d.fillRect(0, 0, width, height);
        // 抗锯齿
        g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        // 画干扰圆圈
        // 设置透明度
        g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.1f * RandomUtil.randomInt(10)));
        InterferingUtil.drawOval(2, g2d, width, height);
        // 画干扰线
        // 设置透明度
        g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.7f));
        g2d.setStroke(new BasicStroke(1.2f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL));
        g2d.setColor(fontColor[0]);
        CubicCurve2D shape = new CubicCurve2D.Double(besselXY[0][0], besselXY[0][1], besselXY[1][0], besselXY[1][1],
                besselXY[2][0], besselXY[2][1], besselXY[3][0], besselXY[3][1]);
        g2d.draw(shape);
        // 画验证码
        g2d.setFont(font);
        FontMetrics fontMetrics = g2d.getFontMetrics();
        // 每一个字符所占的宽度
        int fW = width / strs.length;
        // 字符的左右边距
        int fSp = (fW - (int) fontMetrics.getStringBounds("W", g2d).getWidth()) / 2;
        for (int i = 0; i < strs.length; i++) {
            // 设置透明度
            AlphaComposite ac3 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, getAlpha(flag, i, strs.length));
            g2d.setComposite(ac3);
            g2d.setColor(fontColor[i]);
            // 文字的纵坐标
            int fY = height
                    - ((height - (int) fontMetrics.getStringBounds(String.valueOf(strs[i]), g2d).getHeight()) >> 1);
            g2d.drawString(String.valueOf(strs[i]), i * fW + fSp + 3, fY - 3);
        }
        g2d.dispose();
        return image;
    }

    /**
     * 获取透明度,从0到1,自动计算步长
     *
     * @param i
     * @param j
     * @return 透明度
     */
    public float getAlpha(int i, int j) {
        int num = i + j;
        float r = (float) 1 / (len - 1);
        float s = len * r;
        return num >= len ? (num * r - s) : num * r;
    }

    public float getAlpha(int i, int j, int leng) {
        leng = Math.max(leng, len);
        int num = i + j;
        float r = (float) 1 / (leng - 1);
        float s = leng * r;
        return num >= leng ? (num * r - s) : num * r;
    }

    public static int num(int min, int max) {
        return min + RandomUtil.randomInt(max - min);
    }

    /**
     * 产生0-num的随机数,不包括num
     *
     * @param num 最大值
     * @return 随机数
     */
    public static int num(int num) {
        return RandomUtil.randomInt(num);
    }
}
