package cn.allbs.influx;

import cn.hutool.core.bean.BeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.influxdb.BatchOptions;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.*;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * 功能:
 *
 * @author ChenQi
 * @version 1.0
 * @since 2021/3/5 9:20
 */
@Slf4j
public class InfluxTemplate {

    private final String database;
    private final String retentionPolicy;
    private final String retentionPolicyTime;
    private InfluxDB influxdb;
    private final BatchOptions batchOptions;

    public InfluxTemplate(InfluxDbProperties influxDbProperties, BatchOptions batchOptions) {
        this.batchOptions = batchOptions == null ? BatchOptions.DEFAULTS : batchOptions;
        this.database = influxDbProperties.getDatabase();
        influxDbProperties.setRetentionPolicy(Optional.of(influxDbProperties).map(InfluxDbProperties::getRetentionPolicy).orElse("autogen"));
        influxDbProperties.setRetentionPolicyTime(Optional.of(influxDbProperties).map(InfluxDbProperties::getRetentionPolicyTime).orElse("0"));
        retentionPolicy = influxDbProperties.getRetentionPolicy();
        retentionPolicyTime = influxDbProperties.getRetentionPolicyTime();
        this.influxdb = buildInfluxDb(influxDbProperties);
    }

    public InfluxDB buildInfluxDb(InfluxDbProperties influxDbProperties) {
        if (influxdb == null) {
            influxdb = InfluxDBFactory.connect(influxDbProperties.getOpenUrl(), influxDbProperties.getUsername(), influxDbProperties.getPassword());
            try {
                createDatabase(this.database);
            } catch (Exception e) {
                log.info("create database error " + e.getMessage());
            }
            influxdb.setDatabase(this.database);
        }
        log.debug("init influxDb, current configuration is " + BeanUtil.beanToMap(influxDbProperties));
        return influxdb;
    }

    /**
     * 设置数据保存策略:retentionPolicy策略名 /database 数据库名/ DURATION 数据保存时限/REPLICATION副本个数/结尾 DEFAULT
     * DEFAULT表示设为默认的策略
     */
    public void createRetentionPolicy() {
        String command = String.format("CREATE RETENTION POLICY \"%s\" ON \"%s\" DURATION %s REPLICATION %s DEFAULT",
                retentionPolicy, database, retentionPolicyTime, 1);
        this.query(command);
    }

    /**
     * 设置自定义保留策略
     *
     * @param policyName
     * @param duration
     * @param replication
     * @param isDefault
     */
    public void createRetentionPolicy(String policyName, String duration, int replication, boolean isDefault) {
        String command = String.format("CREATE RETENTION POLICY \"%s\" ON \"%s\" DURATION %s REPLICATION %s ", policyName,
                database, duration, replication);
        if (isDefault) {
            command = command + " DEFAULT";
        }
        this.query(command);
    }

    /**
     * 创建数据库
     *
     * @param database
     */
    public void createDatabase(String database) {
        influxdb.query(new Query("CREATE DATABASE " + database));
    }

    /**
     * 操作数据库
     *
     * @param command
     * @return
     */
    public QueryResult query(String command) {
        return influxdb.query(new Query(command, database));
    }

    /**
     * 插入数据库
     *
     * @param measurement 表名
     * @param tags        tag set
     * @param fields      field set
     */
    public void insert(String measurement, Map<String, String> tags, Map<String, Object> fields) {
        insert(measurement, tags, fields, 0, null);
    }

    /**
     * 方法功能: 单条插入并指定时间戳
     *
     * @param measurement 表名
     * @param tags        tag set
     * @param fields      field set
     * @param time        时间戳
     * @param timeUnit    时间戳的单位
     * @since 2021/3/5 9:32
     */
    public void insert(String measurement, Map<String, String> tags, Map<String, Object> fields, long time, TimeUnit timeUnit) {
        if (time == 0) {
            time = Instant.now().toEpochMilli();
            timeUnit = TimeUnit.MILLISECONDS;
        }
        Point point = Point.measurement(measurement).time(time, timeUnit).tag(tags).fields(fields).build();
        log.info(("influxDB insert data:" + point.toString()));
        influxdb.write(database, retentionPolicy, point);
    }

    /**
     * tag 一定情况下的批量插入
     *
     * @param measurement
     * @param tags
     * @param fieldLists
     */
    public void batchInsert(String measurement, Map<String, String> tags, List<Map<String, Object>> fieldLists) {
        batchInsert(measurement, tags, fieldLists, 0, null);
    }

    /**
     * tag 一定情况下的批量插入
     *
     * @param measurement 表名
     * @param tags        tag
     * @param fieldLists  field
     * @param time        时间戳
     * @param timeUnit    时间戳单位
     */
    public void batchInsert(String measurement, Map<String, String> tags, List<Map<String, Object>> fieldLists, long time, TimeUnit timeUnit) {
        if (time == 0) {
            time = Instant.now().toEpochMilli();
            timeUnit = TimeUnit.MILLISECONDS;
        }
        BatchPoints batchPoints = BatchPoints.database(database).retentionPolicy(retentionPolicy).consistency(InfluxDB.ConsistencyLevel.ALL).build();
        fieldLists.stream().forEach(a -> {
            Point point = Point.measurement(measurement).tag(tags).fields(a).build();
            batchPoints.point(point);
        });
        log.info(("influxDB insert batch data:" + batchPoints.toString()));
        influxdb.write(batchPoints);
    }

    /**
     * 方法功能: 多库多表多条数据插入
     *
     * @param batchPoints
     * @since 2021/3/5 9:35
     */
    public void batchInsert(BatchPoints batchPoints) {
        influxdb.write(batchPoints);
    }

    /**
     * 批量操作结束时手动刷新数据
     */
    public void flush() {
        if (influxdb != null) {
            influxdb.flush();
        }
    }

    /**
     * 如果调用了enableBatch,操作结束时必须调用disableBatch或者手动flush
     */
    public void enableBatch() {
        if (influxdb != null) {
            influxdb.enableBatch(this.batchOptions);
        }
    }

    public void disableBatch() {
        if (influxdb != null) {
            influxdb.disableBatch();
        }
    }

    /**
     * 测试是否已正常连接
     *
     * @return boolean
     */
    public boolean ping() {
        boolean isConnected = false;
        Pong pong;
        try {
            pong = influxdb.ping();
            if (pong != null) {
                isConnected = true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return isConnected;
    }
}
