package cn.allbs.minio.template;

import cn.allbs.common.constant.AllbsCoreConstants;
import cn.allbs.common.constant.StringPool;
import cn.allbs.minio.MinioProperties;
import cn.allbs.minio.model.MinioItem;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.StrUtil;
import io.minio.*;
import io.minio.messages.Bucket;
import io.minio.messages.DeleteObject;
import io.minio.messages.Item;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author : chenqi
 * @version 1.0
 * @since : 2020/8/19 11:19
 */
@Slf4j
public class MinioTemplate {

    private final MinioClient client;

    private final String BASE_BUCKET;

    public MinioTemplate(MinioProperties minioProperties) {
        this.client = MinioClient.builder()
                .endpoint(minioProperties.getUrl(), minioProperties.getPort(), minioProperties.getSecure())
                .credentials(minioProperties.getAccessKey(), minioProperties.getSecretKey())
                .build();
        BASE_BUCKET = minioProperties.getBucketName();
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            createBucket(BASE_BUCKET);
        }
        log.debug(AllbsCoreConstants.ALLBS_TIP + "init minio client, current configuration is " + BeanUtil.beanToMap(minioProperties));
    }

    /**
     * 创建bucket
     *
     * @param bucketName bucket名称
     */
    @SneakyThrows
    public void createBucket(String bucketName) {
        if (StrUtil.isNotEmpty(bucketName) && !client.bucketExists(BucketExistsArgs.builder().bucket(bucketName).build())) {
            MakeBucketArgs args = MakeBucketArgs.builder().bucket(bucketName).build();
            client.makeBucket(args);
        }
    }

    /**
     * 获取全部bucket
     * <p>
     * https://docs.minio.io/cn/java-client-api-reference.html#listBuckets
     */
    @SneakyThrows
    public List<Bucket> getAllBuckets() {
        return client.listBuckets();
    }

    /**
     * @param bucketName bucket名称
     */
    @SneakyThrows
    public Optional<Bucket> getBucket(String bucketName) {
        return client.listBuckets().stream().filter(b -> b.name().equals(bucketName)).findFirst();
    }

    /**
     * @param bucketName bucket名称
     */
    @SneakyThrows
    public void removeBucket(String bucketName) {
        client.removeBucket(RemoveBucketArgs.builder().bucket(bucketName).build());
    }

    /**
     * 根据文件前置查询文件
     *
     * @param bucketName bucket名称
     * @param prefix     前缀
     * @param recursive  是否递归查询
     * @return MinioItem 列表
     */
    @SneakyThrows
    public List<MinioItem> getAllObjectsByPrefix(String bucketName, String prefix, boolean recursive) {
        List<MinioItem> objectList = new ArrayList<>();
        Iterable<Result<Item>> objectsIterator = client
                .listObjects(ListObjectsArgs.builder().bucket(bucketName).prefix(prefix).recursive(recursive).build());

        for (Result<Item> itemResult : objectsIterator) {
            objectList.add(new MinioItem(itemResult.get()));
        }
        return objectList;
    }

    /**
     * 获取文件外链
     *
     * @param bucketName bucket名称
     * @param objectName 文件名称
     * @param expires    过期时间 <=7
     * @return url
     */
    @SneakyThrows
    public String getObjectURL(String bucketName, String objectName, Integer expires) {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            objectName = bucketName + StringPool.SLASH + objectName;
            bucketName = BASE_BUCKET;
        }
        GetPresignedObjectUrlArgs args = GetPresignedObjectUrlArgs.builder()
                .bucket(bucketName)
                .object(objectName)
                .expiry(expires)
                .build();
        return client.getPresignedObjectUrl(args);
    }

    /**
     * 获取文件
     *
     * @param bucketName bucket名称
     * @param objectName 文件名称
     * @return 二进制流
     */
    @SneakyThrows
    public InputStream getObject(String bucketName, String objectName) {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            objectName = bucketName + StringPool.SLASH + objectName;
            bucketName = BASE_BUCKET;
        }
        GetObjectArgs args = GetObjectArgs.builder()
                .bucket(bucketName)
                .object(objectName)
                .build();
        return client.getObject(args);
    }

    /**
     * 上传文件
     *
     * @param bucketName bucket名称
     * @param objectName 文件名称
     * @param stream     文件流
     * @throws Exception https://docs.minio.io/cn/java-client-api-reference.html#putObject
     */
    public void putObject(String bucketName, String objectName, InputStream stream) throws Exception {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            objectName = bucketName + StringPool.SLASH + objectName;
            bucketName = BASE_BUCKET;
        } else {
            createBucket(bucketName);
        }
        PutObjectArgs args = PutObjectArgs.builder()
                .bucket(bucketName)
                .object(objectName)
                .stream(stream, stream.available(), -1)
                .contentType("application/octet-stream")
                .sse(null)
                .build();
        client.putObject(args);
    }

    /**
     * 上传文件
     *
     * @param bucketName  bucket名称
     * @param objectName  文件名称
     * @param stream      文件流
     * @param size        大小
     * @param contextType 类型
     * @throws Exception https://docs.minio.io/cn/java-client-api-reference.html#putObject
     */
    public void putObject(String bucketName, String objectName, InputStream stream, long size, String contextType) throws Exception {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            objectName = bucketName + StringPool.SLASH + objectName;
            bucketName = BASE_BUCKET;
        } else {
            createBucket(bucketName);
        }
        PutObjectArgs args = PutObjectArgs.builder()
                .bucket(bucketName)
                .object(objectName)
                .stream(stream, size, -1)
                .contentType(contextType)
                .sse(null)
                .build();
        client.putObject(args);
    }

    /**
     * 获取文件信息
     *
     * @param bucketName bucket名称
     * @param objectName 文件名称
     * @throws Exception https://docs.minio.io/cn/java-client-api-reference.html#statObject
     */
    public StatObjectResponse getObjectInfo(String bucketName, String objectName) throws Exception {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            objectName = bucketName + StringPool.SLASH + objectName;
            bucketName = BASE_BUCKET;
        }
        StatObjectArgs args = StatObjectArgs.builder().bucket(bucketName).object(objectName).build();
        return client.statObject(args);
    }

    /**
     * 删除文件
     *
     * @param bucketName bucket名称
     * @param objectName 文件名称
     * @throws Exception https://docs.minio.io/cn/java-client-api-reference.html#removeObject
     */
    public void removeObject(String bucketName, String objectName) throws Exception {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            objectName = bucketName + StringPool.SLASH + objectName;
            bucketName = BASE_BUCKET;
        }
        RemoveObjectArgs args = RemoveObjectArgs.builder().bucket(bucketName).object(objectName).build();
        client.removeObject(args);
    }

    /**
     * 同一个桶内批量删除文件
     *
     * @param bucketName     bucket名称
     * @param objectNameList 文件名称list
     * @throws Exception https://docs.minio.io/cn/java-client-api-reference.html#removeObject
     */
    public void removeObjects(String bucketName, List<String> objectNameList) throws Exception {
        if (StrUtil.isNotEmpty(BASE_BUCKET)) {
            List<String> deleteObjectNameList = new ArrayList<>();
            objectNameList.forEach(a -> deleteObjectNameList.add(bucketName + StringPool.SLASH + a));
            List<DeleteObject> list = deleteObjectNameList.stream().map(DeleteObject::new).collect(Collectors.toList());
            RemoveObjectsArgs removeObjectsArgs = RemoveObjectsArgs.builder().bucket(BASE_BUCKET).objects(list).build();
            client.removeObjects(removeObjectsArgs);
            return;
        }
        List<DeleteObject> list = objectNameList.stream().map(DeleteObject::new).collect(Collectors.toList());
        RemoveObjectsArgs removeObjectsArgs = RemoveObjectsArgs.builder().bucket(bucketName).objects(list).build();
        client.removeObjects(removeObjectsArgs);
    }
}
