package cn.allbs.sms.huawei;

import cn.hutool.core.bean.BeanUtil;
import com.google.gson.Gson;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 华为云短信发送工具类
 * 如果JDK版本是1.8,可使用原生Base64类
 *
 * @author chenqi
 */
@AllArgsConstructor
@Slf4j
public class HuaweiSmsTemplate {

    /**
     * 无需修改,用于格式化鉴权头域,给"X-WSSE"参数赋值
     */
    private static final String WSSE_HEADER_FORMAT = "UsernameToken Username=\"%s\",PasswordDigest=\"%s\",Nonce=\"%s\",Created=\"%s\"";
    /**
     * 无需修改,用于格式化鉴权头域,给"Authorization"参数赋值
     */
    private static final String AUTH_HEADER_VALUE = "WSSE realm=\"SDP\",profile=\"UsernameToken\",type=\"Appkey\"";

    public static final String SMS_CODE = "code";
    public static final String SMS_OK = "000000";

    private String url;
    private String appKey;
    private String appSecret;
    private String sender;
    private String signature;

    /**
     * 发送短信
     *
     * @param templateId 模板Id
     * @param receivers  接收者的手机号，多个用字符串分割
     * @param params     短信模板参数数组
     * @return 是否发送成功
     */
    public Boolean sendSms(String templateId, String receivers, String[] params) {
        Boolean sendFlag = false;
        Gson gson = new Gson();
        String templateParas = gson.toJson(params);
        //选填,短信状态报告接收地址,推荐使用域名,为空或者不填表示不接收状态报告
        String statusCallBack = "";
        //请求Body,不携带签名名称时,signature请填null
        String body = buildRequestBody(sender, receivers, templateId, templateParas, statusCallBack, signature);
        if (null == body || body.isEmpty()) {
            log.error("allbs-sms 温馨提示: body is null.");
            return false;
        }

        /*
          选填,使用无变量模板时请赋空值 String templateParas = "";
          单变量模板示例:模板内容为"您的验证码是${1}"时,templateParas可填写为"[\"369751\"]"
          双变量模板示例:模板内容为"您有${1}件快递请到${2}领取"时,templateParas可填写为"[\"3\",\"人民公园正门\"]"
          模板中的每个变量都必须赋值，且取值不能为空
          查看更多模板和变量规范:产品介绍>模板和变量规范
         */

        //请求Headers中的X-WSSE参数值
        String wsseHeader = buildWsseHeader(appKey, appSecret);
        if (null == wsseHeader || wsseHeader.isEmpty()) {
            log.error("allbs-sms 温馨提示: wsse header is null.");
            return false;
        }
        //如果JDK版本是1.8,可使用如下代码
        //为防止因HTTPS证书认证失败造成API调用失败,需要先忽略证书信任问题
        CloseableHttpClient client = null;
        try {
            client = HttpClients.custom()
                    .setSSLContext(new SSLContextBuilder().loadTrustMaterial(null,
                            (x509CertChain, authType) -> true).build())
                    .setSSLHostnameVerifier(NoopHostnameVerifier.INSTANCE)
                    .build();
        } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
            e.printStackTrace();
        }
        HttpResponse response = null;
        try {
            //请求方法POST
            assert client != null;
            response = client.execute(RequestBuilder.create("POST")
                    .setUri(url)
                    .addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded")
                    .addHeader(HttpHeaders.AUTHORIZATION, AUTH_HEADER_VALUE)
                    .addHeader("X-WSSE", wsseHeader)
                    .setEntity(new StringEntity(body)).build());
            //打印响应头域信息
            log.info("allbs-sms 温馨提示:  response is" + response.toString());
            //打印响应消息实体
            HttpEntity entity = response.getEntity();
            String result = EntityUtils.toString(entity, "UTF-8");
            if (BeanUtil.isNotEmpty(entity)) {
                Map<String, Object> resultMap = gson.fromJson(result, Map.class);
                log.info("allbs-sms 温馨提示: the message is to be sent " + resultMap);
                if (resultMap.get(SMS_CODE).equals(SMS_OK)) {
                    log.info("allbs-sms 温馨提示: 短信发送成功,message content is" + result);
                }
            }

        } catch (IOException e) {
            e.printStackTrace();
            log.error("allbs-sms 温馨提示: 短信API调用失败！");
        }
        return true;
    }


    /**
     * 构造请求Body体
     *
     * @param sender            发送方
     * @param receiver          接收方
     * @param templateId        模板id
     * @param templateParas     模板参数
     * @param statusCallbackUrl 回调url
     * @param signature         | 签名名称,使用国内短信通用模板时填写
     * @return 请求连接
     */
    static String buildRequestBody(String sender, String receiver, String templateId, String templateParas,
                                   String statusCallbackUrl, String signature) {
        if (null == sender || null == receiver || null == templateId || sender.isEmpty() || receiver.isEmpty()
                || templateId.isEmpty()) {
            log.error("allbs-sms 温馨提示: buildRequestBody sender, receiver or templateId is null.");
            return null;
        }
        List<NameValuePair> keyValues = new ArrayList<NameValuePair>();

        keyValues.add(new BasicNameValuePair("from", sender));
        keyValues.add(new BasicNameValuePair("to", receiver));
        keyValues.add(new BasicNameValuePair("templateId", templateId));
        if (null != templateParas && !templateParas.isEmpty()) {
            keyValues.add(new BasicNameValuePair("templateParas", templateParas));
        }
        if (null != statusCallbackUrl && !statusCallbackUrl.isEmpty()) {
            keyValues.add(new BasicNameValuePair("statusCallback", statusCallbackUrl));
        }
        if (null != signature && !signature.isEmpty()) {
            keyValues.add(new BasicNameValuePair("signature", signature));
        }

        return URLEncodedUtils.format(keyValues, Charset.forName("UTF-8"));
    }

    /**
     * 构造X-WSSE参数值
     *
     * @param appKey appKey
     * @param appSecret appSecret
     * @return 请求连接
     */
    static String buildWsseHeader(String appKey, String appSecret) {
        if (null == appKey || null == appSecret || appKey.isEmpty() || appSecret.isEmpty()) {
            log.error("allbs-sms 温馨提示: appKey or appSecret is null in method buildWsseHeader");
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        //Created
        String time = sdf.format(new Date());
        //Nonce
        String nonce = UUID.randomUUID().toString().replace("-", "");

        byte[] passwordDigest = DigestUtils.sha256(nonce + time + appSecret);
        String hexDigest = Hex.encodeHexString(passwordDigest);

        //如果JDK版本是1.8,请加载原生Base64类,并使用如下代码
        //PasswordDigest
        String passwordDigestBase64Str = Base64.getEncoder().encodeToString(hexDigest.getBytes());
        return String.format(WSSE_HEADER_FORMAT, appKey, passwordDigestBase64Str, nonce, time);
    }
}
