package cn.bestwu.framework.rest.aspect;

import cn.bestwu.framework.event.LogEvent;
import cn.bestwu.framework.rest.support.Log;
import cn.bestwu.framework.rest.support.PrincipalNamePutEvent;
import cn.bestwu.framework.rest.support.RequestJsonViewResponseBodyAdvice;
import cn.bestwu.framework.rest.support.Resource;
import cn.bestwu.framework.util.ResourceUtil;
import cn.bestwu.framework.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.http.HttpHeaders;
import org.springframework.http.converter.json.MappingJacksonValue;
import org.springframework.http.server.ServletServerHttpRequest;

import javax.servlet.RequestDispatcher;
import javax.servlet.http.HttpServletRequest;
import java.util.Map;

@Aspect
@Slf4j
public class LogAspect {
	private final String PUT_PARAMETER_MAP = "PUT_PARAMETER_MAP";
	private final String PRINCIPAL_NAME = "PRINCIPAL_NAME";

	protected ApplicationEventPublisher publisher;

	@Autowired(required = false)
	protected HttpServletRequest request;

	public LogAspect(ApplicationEventPublisher publisher) {
		this.publisher = publisher;
	}

	/**
	 * 处理有异常时，跳转异常处理controller时，出现的用户信息丢失，及可能的参数丢失。
	 */
	@AfterThrowing(value = "@annotation(org.springframework.web.bind.annotation.RequestMapping)")
	public void afterThrowing() {
		Resource<String> source = new Resource<>();
		publisher.publishEvent(new PrincipalNamePutEvent(source));
		request.setAttribute(PRINCIPAL_NAME, source.getContent());

		if ("PUT".equals(request.getMethod()))
			request.setAttribute(PUT_PARAMETER_MAP, request.getParameterMap());
	}

	@Autowired(required = false)
	private RequestJsonViewResponseBodyAdvice requestJsonViewResponseBodyAdvice;

	@SuppressWarnings("unchecked")
	@AfterReturning(value = "@annotation(org.springframework.web.bind.annotation.RequestMapping)", returning = "result")
	public void log(Object result) {
		if (log.isInfoEnabled()) {
			// request信息
			String ipAddress = request.getRemoteAddr();
			String servletPath = (String) request.getAttribute(RequestDispatcher.FORWARD_SERVLET_PATH);
			if (servletPath == null) {
				servletPath = request.getServletPath();
			}
			String principalName = (String) request.getAttribute(PRINCIPAL_NAME);
			if (principalName == null) {
				Resource<String> source = new Resource<>();
				publisher.publishEvent(new PrincipalNamePutEvent(source));
				principalName = source.getContent();
			}

			String requestMethod = request.getMethod();
			Map<String, String[]> parameterMap = null;
			try {
				if ("PUT".equals(requestMethod)) {
					parameterMap = (Map<String, String[]>) request.getAttribute(PUT_PARAMETER_MAP);
				}
				if (parameterMap == null) {
					parameterMap = request.getParameterMap();
				}
			} catch (Exception e) {
				if (log.isDebugEnabled()) {
					log.debug("获取请求参数出错", e);
				}
				parameterMap = null;
			}

			ServletServerHttpRequest servletServerHttpRequest = new ServletServerHttpRequest(request);
			HttpHeaders headers = servletServerHttpRequest.getHeaders();
			String MSG_CODE = "{} 的 [{}] {} {} \nrequest headers:\n{} \nrequest parameters:\n{} \nresponse:\n{}";

			principalName = principalName == null ? (request.getRemoteUser() == null ? "anonymousUser" : request.getRemoteUser()) : principalName;
			String requestSignature = ResourceUtil.getRequestSignature(request);
			if (log.isDebugEnabled()) {
				String resultStr;
				if ("get_logs_index".equals(requestSignature)) {
					resultStr = StringUtil.subString(result.toString(), 100);
				} else {
					if (requestJsonViewResponseBodyAdvice != null) {
						MappingJacksonValue mappingJacksonValue = new MappingJacksonValue(result);
						requestJsonViewResponseBodyAdvice.beforeBodyWrite(mappingJacksonValue, request);
						result = mappingJacksonValue;
					}
					resultStr = StringUtil.valueOf(result, true);
				}
				log.info(MSG_CODE, ipAddress, principalName, requestMethod,
						servletPath, StringUtil.valueOf(headers, true), StringUtil.valueOf(parameterMap, true),
						resultStr);
			} else
				log.info(MSG_CODE, ipAddress, principalName, requestMethod,
						servletPath, StringUtil.valueOf(headers, true), StringUtil.subString(StringUtil.valueOf(parameterMap), 100),
						StringUtil.subString(String.valueOf(result), 100));

			{
				Log log = new Log();
				log.setIpAddress(ipAddress);
				log.setParameters(StringUtil.valueOf(parameterMap, true));
				log.setRequestMethod(requestMethod);
				log.setRequestSignature(requestSignature);
				log.setServletPath(servletPath);
				log.setRequestHeaders(StringUtil.valueOf(headers, true));
				log.setPrincipalName(principalName);

				publisher.publishEvent(new LogEvent(log));
			}
		}
	}

}
