package cn.bestwu.security.oauth2.social.provider;

import cn.bestwu.security.oauth2.OAuth2AccessDeniedHandler;
import cn.bestwu.security.oauth2.OAuth2AuthenticationEntryPoint;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.security.oauth2.OAuth2ClientProperties;
import org.springframework.boot.autoconfigure.security.oauth2.authserver.AuthorizationServerProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.oauth2.config.annotation.builders.ClientDetailsServiceBuilder;
import org.springframework.security.oauth2.config.annotation.builders.InMemoryClientDetailsServiceBuilder;
import org.springframework.security.oauth2.config.annotation.configurers.ClientDetailsServiceConfigurer;
import org.springframework.security.oauth2.config.annotation.web.configuration.AuthorizationServerConfigurerAdapter;
import org.springframework.security.oauth2.config.annotation.web.configuration.AuthorizationServerEndpointsConfiguration;
import org.springframework.security.oauth2.config.annotation.web.configurers.AuthorizationServerEndpointsConfigurer;
import org.springframework.security.oauth2.config.annotation.web.configurers.AuthorizationServerSecurityConfigurer;
import org.springframework.security.oauth2.provider.CompositeTokenGranter;
import org.springframework.security.oauth2.provider.TokenGranter;
import org.springframework.security.oauth2.provider.client.BaseClientDetails;
import org.springframework.security.oauth2.provider.token.TokenStore;

import javax.annotation.PostConstruct;
import java.util.*;

/**
 * 自动配置
 *
 * @param <T> 系统用户
 */
@Configuration
@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE + 10)
@Order(Ordered.HIGHEST_PRECEDENCE)
@ConditionalOnBean(AuthorizationServerEndpointsConfiguration.class)
@EnableConfigurationProperties({ AuthorizationServerProperties.class, SocialProperties.class })
public class SocialAuthorizationServerConfiguration<T> extends AuthorizationServerConfigurerAdapter {
	private static final Log logger = LogFactory.getLog(SocialAuthorizationServerConfiguration.class);

	@Autowired
	private BaseClientDetails details;

	@Autowired
	private AuthenticationManager authenticationManager;

	@Autowired(required = false)
	private TokenStore tokenStore;

	@Autowired
	private AuthorizationServerProperties properties;

	@Autowired
	private SocialAdapter<T> socialAdapter;
	@Autowired
	private AuthorizationServerEndpointsConfiguration serverEndpointsConfiguration;

	@Bean
	public SignupEndpoint<T> signupEndpoint() {
		AuthorizationServerEndpointsConfigurer endpointsConfigurer = serverEndpointsConfiguration.getEndpointsConfigurer();
		return new SignupEndpoint<>(socialAdapter, endpointsConfigurer.getTokenServices(), endpointsConfigurer.getExceptionTranslator());
	}

	@Override
	public void configure(AuthorizationServerEndpointsConfigurer endpoints) throws Exception {
		if (this.tokenStore != null) {
			endpoints.tokenStore(this.tokenStore);
		}
		if (this.details.getAuthorizedGrantTypes().contains("password")) {
			endpoints.authenticationManager(this.authenticationManager);
		}

		TokenGranter granter = endpoints.getTokenGranter();

		List<TokenGranter> tokenGranters = new ArrayList<>();
		tokenGranters.add(granter);

		tokenGranters.add(new SocialTokenGranter<>(endpoints.getTokenServices(), endpoints.getClientDetailsService(), socialAdapter));

		TokenGranter tokenGranter = new CompositeTokenGranter(tokenGranters);

		endpoints.tokenGranter(tokenGranter);
	}

	@Override
	public void configure(ClientDetailsServiceConfigurer clients) throws Exception {
		ClientDetailsServiceBuilder<InMemoryClientDetailsServiceBuilder>.ClientBuilder builder = clients
				.inMemory().withClient(this.details.getClientId());
		builder.secret(this.details.getClientSecret())
				.resourceIds(this.details.getResourceIds().toArray(new String[0]))
				.authorizedGrantTypes(this.details.getAuthorizedGrantTypes().toArray(new String[0]))
				.authorities(AuthorityUtils.authorityListToSet(this.details.getAuthorities()).toArray(new String[0]))
				.scopes(this.details.getScope().toArray(new String[0]));

		if (this.details.getAutoApproveScopes() != null) {
			builder.autoApprove(this.details.getAutoApproveScopes().toArray(new String[0]));
		}
		if (this.details.getAccessTokenValiditySeconds() != null) {
			builder.accessTokenValiditySeconds(this.details.getAccessTokenValiditySeconds());
		}
		if (this.details.getRefreshTokenValiditySeconds() != null) {
			builder.refreshTokenValiditySeconds(this.details.getRefreshTokenValiditySeconds());
		}
		if (this.details.getRegisteredRedirectUri() != null) {
			builder.redirectUris(this.details.getRegisteredRedirectUri().toArray(new String[0]));
		}
	}

	@Override
	public void configure(AuthorizationServerSecurityConfigurer security)
			throws Exception {
		if (this.properties.getCheckTokenAccess() != null) {
			security.checkTokenAccess(this.properties.getCheckTokenAccess());
		}
		if (this.properties.getTokenKeyAccess() != null) {
			security.tokenKeyAccess(this.properties.getTokenKeyAccess());
		}
		if (this.properties.getRealm() != null) {
			security.realm(this.properties.getRealm());
		}

		security.accessDeniedHandler(new OAuth2AccessDeniedHandler());
		security.authenticationEntryPoint(new OAuth2AuthenticationEntryPoint());
	}

	@Configuration
	protected static class ClientDetailsLogger {

		@Autowired
		private OAuth2ClientProperties credentials;

		@PostConstruct
		public void init() {
			String prefix = "security.oauth2.client";
			boolean defaultSecret = this.credentials.isDefaultSecret();
			logger.info(String.format("Initialized OAuth2 Client\n\n%s.clientId = %s\n%s.secret = %s\n\n",
					prefix, this.credentials.getClientId(), prefix,
					defaultSecret ? this.credentials.getClientSecret() : "****"));
		}

	}

	@Configuration
	@ConditionalOnMissingBean(BaseClientDetails.class)
	protected static class BaseClientDetailsConfiguration {

		@Autowired
		private OAuth2ClientProperties client;

		@Bean
		@ConfigurationProperties("security.oauth2.client")
		public BaseClientDetails oauth2ClientDetails() {
			BaseClientDetails details = new BaseClientDetails();
			if (this.client.getClientId() == null) {
				this.client.setClientId(UUID.randomUUID().toString());
			}
			details.setClientId(this.client.getClientId());
			details.setClientSecret(this.client.getClientSecret());
			details.setAuthorizedGrantTypes(
					Arrays.asList("authorization_code", "password", "client_credentials", "implicit", "refresh_token", SocialAdapter.WEIXIN, SocialAdapter.WEIBO, SocialAdapter.QQ));
			details.setAuthorities(AuthorityUtils.commaSeparatedStringToAuthorityList("ROLE_USER"));
			details.setRegisteredRedirectUri(Collections.emptySet());
			return details;
		}

	}

}