package cn.boboweike.carrot.configuration;

import cn.boboweike.carrot.server.BackgroundTaskServer;

/**
 * This class provides the entry point for the Carrot configuration. This is needed when you want to use the static methods
 * on {@link cn.boboweike.carrot.scheduling.BackgroundTask} to enqueue and schedule tasks. It also allows to startup the Dashboard which
 * will be available on port 8000.
 *
 * <h3>An example:</h3>
 * <pre>
 *      Carrot.configure()
 *                 .useTaskActivator(taskActivator)
 *                 .useTaskStorageProvider(taskStorageProvider)
 *                 .useBackgroundTaskServer()
 *                 .useJmxExtensions()
 *                 .useDashboard()
 *                 .initialize();
 * </pre>
 * <p>
 * Both the backgroundTaskServer and the dashboard fluent API allows to be enabled or disabled using ENV variables which
 * is handy in docker containers.
 * <h3>An example:</h3>
 * <pre>
 *     boolean isBackgroundTaskServerEnabled = true; // or get it via ENV variables
 *     boolean isDashboardEnabled = true; // or get it via ENV variables
 *     Carrot.configure()
 *                 .useTaskStorageProvider(taskStorageProvider)
 *                 .useTaskActivator(taskActivator)
 *                 .useBackgroundTaskServerIf(isBackgroundTaskServerEnabled)
 *                 .useDashboardIf(isDashboardEnabled)
 *                 .useJmxExtensions()
 *                 .initialize();
 * </pre>
 */
public class Carrot {
    private static CarrotConfiguration carrotConfiguration;

    private Carrot() {
    }

    public static CarrotConfiguration configure() {
        carrotConfiguration = new CarrotConfiguration();
        Runtime.getRuntime().addShutdownHook(new Thread(Carrot::destroy, "extShutdownHook"));
        return carrotConfiguration;
    }

    public static CarrotConfiguration destroy() {
        if (carrotConfiguration != null) {
            if (carrotConfiguration.backgroundTaskServer != null) carrotConfiguration.backgroundTaskServer.stop();
            if (carrotConfiguration.dashboardWebServer != null) carrotConfiguration.dashboardWebServer.stop();
            if (carrotConfiguration.storageProvider != null) carrotConfiguration.storageProvider.close();
            if (carrotConfiguration.microMeterIntegration != null) carrotConfiguration.microMeterIntegration.close();
        }
        return carrotConfiguration;
    }

    public static BackgroundTaskServer getBackgroundTaskServer() {
        if (carrotConfiguration == null)
            throw new IllegalStateException("You don't seem to use the Fluent API. This method is only available if you are using the Fluent API to configure Carrot");
        if (carrotConfiguration.backgroundTaskServer == null)
            throw new IllegalStateException("The background task server is not configured. Are you perhaps only running the TaskScheduler or the Dashboard on this server instance?");
        return carrotConfiguration.backgroundTaskServer;
    }
}
