package cn.boboweike.carrot.configuration;

import cn.boboweike.carrot.dashboard.CarrotDashboardWebServer;
import cn.boboweike.carrot.dashboard.CarrotDashboardWebServerConfiguration;
import cn.boboweike.carrot.scheduling.TaskRequestScheduler;
import cn.boboweike.carrot.scheduling.TaskScheduler;
import cn.boboweike.carrot.scheduling.partition.Partitioner;
import cn.boboweike.carrot.server.BackgroundTaskServer;
import cn.boboweike.carrot.server.BackgroundTaskServerConfiguration;
import cn.boboweike.carrot.server.TaskActivator;
import cn.boboweike.carrot.server.jmx.CarrotJMXExtensions;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.details.CachingTaskDetailsGenerator;
import cn.boboweike.carrot.tasks.details.TaskDetailsGenerator;
import cn.boboweike.carrot.tasks.filters.TaskFilter;
import cn.boboweike.carrot.tasks.mappers.TaskMapper;
import cn.boboweike.carrot.utils.mapper.JsonMapper;
import cn.boboweike.carrot.utils.mapper.JsonMapperException;
import cn.boboweike.carrot.utils.mapper.gson.GsonJsonMapper;
import cn.boboweike.carrot.utils.mapper.jackson.JacksonJsonMapper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static cn.boboweike.carrot.dashboard.CarrotDashboardWebServerConfiguration.usingStandardDashboardConfiguration;
import static cn.boboweike.carrot.server.BackgroundTaskServerConfiguration.usingStandardBackgroundTaskServerConfiguration;
import static cn.boboweike.carrot.utils.mapper.JsonMapperValidator.validateJsonMapper;
import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.classExists;
import static java.util.Optional.ofNullable;

/**
 * The main class to configure Carrot
 */
public class CarrotConfiguration {
    TaskActivator taskActivator;
    JsonMapper jsonMapper;
    TaskMapper taskMapper;
    final List<TaskFilter> taskFilters;
    TaskDetailsGenerator taskDetailsGenerator;
    PartitionedStorageProvider storageProvider;
    BackgroundTaskServer backgroundTaskServer;
    CarrotDashboardWebServer dashboardWebServer;
    CarrotJMXExtensions jmxExtension;
    CarrotMicroMeterIntegration microMeterIntegration;

    CarrotConfiguration() {
        this.jsonMapper = determineJsonMapper();
        this.taskMapper = new TaskMapper(jsonMapper);
        this.taskDetailsGenerator = new CachingTaskDetailsGenerator();
        this.taskFilters = new ArrayList<>();
    }

    /**
     * The {@link JsonMapper} to transform tasks to json in the database
     *
     * @param jsonMapper the {@link JsonMapper} to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useJsonMapper(JsonMapper jsonMapper) {
        if (this.storageProvider != null) {
            throw new IllegalStateException("Please configure the TaskActivator before the StorageProvider.");
        }
        if (this.dashboardWebServer != null) {
            throw new IllegalStateException("Please configure the TaskActivator before the DashboardWebServer.");
        }
        this.jsonMapper = validateJsonMapper(jsonMapper);
        this.taskMapper = new TaskMapper(jsonMapper);
        return this;
    }

    /**
     * The {@link TaskActivator} is used to resolve tasks from the IoC framework
     *
     * @param taskActivator the {@link TaskActivator} to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useTaskActivator(TaskActivator taskActivator) {
        if (this.backgroundTaskServer != null) {
            throw new IllegalStateException("Please configure the TaskActivator before the BackgroundTaskServer.");
        }
        this.taskActivator = taskActivator;
        return this;
    }

    /**
     * Allows to set the PartitionedStorageProvider that Carrot will use.
     *
     * @param storageProvider the PartitionedStorageProvider to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useStorageProvider(PartitionedStorageProvider storageProvider) {
        this.storageProvider = storageProvider;
        storageProvider.setTaskMapper(taskMapper);
        return this;
    }

    /**
     * Allows to set the Partitioner that Carrot will use.
     * If not configured, the default {@link cn.boboweike.carrot.scheduling.partition.RandomPartitioner} will be used.
     *
     * @param partitioner the Partitioner to use
     * @return the same configuration instance with provides a fluent api
     */
    public CarrotConfiguration usePartitioner(Partitioner partitioner) {
        if (this.storageProvider == null) {
            throw new IllegalStateException("Please configure the PartitionedStorageProvider before the Partitioner.");
        }
        this.storageProvider.setPartitioner(partitioner);
        return this;
    }

    /**
     * Allows setting extra TaskFilters or to provide another implementation of the {@link cn.boboweike.carrot.tasks.filters.RetryFilter}
     *
     * @param taskFilters the taskFilters to use for each task.
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration withTaskFilter(TaskFilter... taskFilters) {
        if (this.backgroundTaskServer != null) {
            throw new IllegalStateException("Please configure the TaskFilters before the BackgroundTaskServer.");
        }
        this.taskFilters.addAll(Arrays.asList(taskFilters));
        return this;
    }

    /**
     * Provides a default {@link BackgroundTaskServer} that is configured using a number of threads depending on the amount of CPU.
     *
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServer() {
        return useBackgroundTaskServerIf(true);
    }

    /**
     * Provides a default {@link BackgroundTaskServer} if the guard is true and that is configured using a number of threads depending on the amount of CPU.
     *
     * @param guard whether to start a BackgroundTaskServer or not.
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServerIf(boolean guard) {
        return useBackgroundTaskServerIf(guard, usingStandardBackgroundTaskServerConfiguration());
    }

    /**
     * Provides a default {@link BackgroundTaskServer} that is configured using a given number of threads.
     *
     * @param workerCount the number of worker threads to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServer(int workerCount) {
        return useBackgroundTaskServerIf(true, workerCount);
    }

    /**
     * Provides a default {@link BackgroundTaskServer} if the guard is true and that is configured using a given number of threads.
     *
     * @param guard       whether to start a BackgroundTaskServer or not.
     * @param workerCount the number of worker threads to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServerIf(boolean guard, int workerCount) {
        return useBackgroundTaskServerIf(guard, usingStandardBackgroundTaskServerConfiguration().andWorkerCount(workerCount));
    }

    /**
     * Provides a default {@link BackgroundTaskServer} that is configured using the given {@link BackgroundTaskServerConfiguration}
     *
     * @param configuration the configuration for the backgroundTaskServer to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServer(BackgroundTaskServerConfiguration configuration) {
        return useBackgroundTaskServerIf(true, configuration);
    }

    /**
     * Provides a default {@link BackgroundTaskServer} that is configured using the given {@link BackgroundTaskServerConfiguration}
     *
     * @param configuration            the configuration for the backgroundTaskServer to use
     * @param startBackgroundTaskServer whether to start the background task server immediately
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServer(BackgroundTaskServerConfiguration configuration, boolean startBackgroundTaskServer) {
        return useBackgroundTaskServerIf(true, configuration, startBackgroundTaskServer);
    }

    /**
     * Provides a default {@link BackgroundTaskServer} if the guard is true and that is configured using the given {@link BackgroundTaskServerConfiguration}
     *
     * @param guard         whether to start a BackgroundTaskServer or not.
     * @param configuration the configuration for the backgroundTaskServer to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServerIf(boolean guard, BackgroundTaskServerConfiguration configuration) {
        return useBackgroundTaskServerIf(guard, configuration, true);
    }

    /**
     * Provides a default {@link BackgroundTaskServer} if the guard is true and that is configured using the given {@link BackgroundTaskServerConfiguration}
     *
     * @param guard                    whether to create a BackgroundTaskServer or not.
     * @param configuration            the configuration for the backgroundTaskServer to use
     * @param startBackgroundTaskServer whether to start the background task server immediately
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useBackgroundTaskServerIf(boolean guard, BackgroundTaskServerConfiguration configuration, boolean startBackgroundTaskServer) {
        if (guard) {
            this.backgroundTaskServer = new BackgroundTaskServer(storageProvider, taskActivator, configuration);
            this.backgroundTaskServer.setTaskFilters(taskFilters);
            this.backgroundTaskServer.start(startBackgroundTaskServer);
        }
        return this;
    }

    /**
     * Provides a dashboard on port 8000
     *
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useDashboard() {
        return useDashboardIf(true);
    }

    /**
     * Provides a dashboard on port 8000 if the guard is true
     *
     * @param guard whether to start a Dashboard or not.
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useDashboardIf(boolean guard) {
        return useDashboardIf(guard, usingStandardDashboardConfiguration());
    }

    /**
     * Provides a dashboard on the given port
     *
     * @param dashboardPort the port on which to start the {@link CarrotDashboardWebServer}
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useDashboard(int dashboardPort) {
        return useDashboardIf(true, dashboardPort);
    }

    /**
     * Provides a dashboard on the given port if the guard is true
     *
     * @param guard         whether to start a Dashboard or not.
     * @param dashboardPort the port on which to start the {@link CarrotDashboardWebServer}
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useDashboardIf(boolean guard, int dashboardPort) {
        return useDashboardIf(guard, usingStandardDashboardConfiguration().andPort(dashboardPort));
    }

    /**
     * Provides a dashboard using the given {@link CarrotDashboardWebServerConfiguration}
     *
     * @param configuration the {@link CarrotDashboardWebServerConfiguration} to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useDashboard(CarrotDashboardWebServerConfiguration configuration) {
        return useDashboardIf(true, configuration);
    }

    /**
     * Provides a dashboard using the given {@link CarrotDashboardWebServerConfiguration} if the guard is true
     *
     * @param guard         whether to start a Dashboard or not.
     * @param configuration the {@link CarrotDashboardWebServerConfiguration} to use
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useDashboardIf(boolean guard, CarrotDashboardWebServerConfiguration configuration) {
        if (guard) {
            this.dashboardWebServer = new CarrotDashboardWebServer(storageProvider, jsonMapper, configuration);
            this.dashboardWebServer.start();
        }
        return this;
    }

    /**
     * If called, this method will register JMX Extensions to monitor Carrot via JMX
     *
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useJmxExtensions() {
        return useJmxExtensionsIf(true);
    }

    /**
     * Enables JMX Extensions to monitor Carrot via JMX if the guard is true
     *
     * @param guard whether to start the JXM Extensions or not.
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useJmxExtensionsIf(boolean guard) {
        if (guard) {
            if (backgroundTaskServer == null)
                throw new IllegalStateException("Please configure the BackgroundTaskServer before the JMXExtension.");
            if (storageProvider == null)
                throw new IllegalStateException("Please configure the StorageProvider before the JMXExtension.");
            this.jmxExtension = new CarrotJMXExtensions(backgroundTaskServer, storageProvider);
        }
        return this;
    }

    /**
     * Allows integrating MicroMeter metrics into Carrot
     *
     * @param microMeterIntegration the CarrotMicroMeterIntegration
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useMicroMeter(CarrotMicroMeterIntegration microMeterIntegration) {
        this.microMeterIntegration = microMeterIntegration;
        return this;
    }

    /**
     * Specifies which {@link TaskDetailsGenerator} to use.
     *
     * @param taskDetailsGenerator the TaskDetailsGenerator to use.
     * @return the same configuration instance which provides a fluent api
     */
    public CarrotConfiguration useTaskDetailsGenerator(TaskDetailsGenerator taskDetailsGenerator) {
        this.taskDetailsGenerator = taskDetailsGenerator;
        return this;
    }

    /**
     * Initializes Carrot and returns a {@link TaskScheduler} which can then be used to register in the IoC framework
     * or to enqueue/schedule some Tasks.
     *
     * @return a TaskScheduler to enqueue/schedule new tasks
     */
    public CarrotConfigurationResult initialize() {
        ofNullable(microMeterIntegration).ifPresent(meterRegistry -> meterRegistry.initialize(storageProvider, backgroundTaskServer));
        final TaskScheduler taskScheduler = new TaskScheduler(storageProvider, taskDetailsGenerator, taskFilters);
        final TaskRequestScheduler taskRequestScheduler = new TaskRequestScheduler(storageProvider, taskFilters);
        return new CarrotConfigurationResult(taskScheduler, taskRequestScheduler);
    }

    private static JsonMapper determineJsonMapper() {
        if (classExists("com.fasterxml.jackson.databind.ObjectMapper")) {
            return new JacksonJsonMapper();
        } else if (classExists("com.google.gson.Gson")) {
            return new GsonJsonMapper();
        } else {
            throw new JsonMapperException("No JsonMapper class is found. Make sure you have either Jackson compliant library available on your classpath");
        }
    }

    public static class CarrotConfigurationResult {

        private final TaskScheduler taskScheduler;
        private final TaskRequestScheduler taskRequestScheduler;

        public CarrotConfigurationResult(TaskScheduler taskScheduler, TaskRequestScheduler taskRequestScheduler) {
            this.taskScheduler = taskScheduler;
            this.taskRequestScheduler = taskRequestScheduler;
        }

        public TaskScheduler getTaskScheduler() {
            return taskScheduler;
        }

        public TaskRequestScheduler getTaskRequestScheduler() {
            return taskRequestScheduler;
        }
    }
}
