package cn.boboweike.carrot.dashboard;

import cn.boboweike.carrot.dashboard.server.http.RestHttpHandler;
import cn.boboweike.carrot.dashboard.server.http.handlers.HttpRequestHandler;
import cn.boboweike.carrot.dashboard.ui.model.RecurringTaskUIModel;
import cn.boboweike.carrot.dashboard.ui.model.VersionUIModel;
import cn.boboweike.carrot.dashboard.ui.model.problems.ProblemsManager;
import cn.boboweike.carrot.storage.*;
import cn.boboweike.carrot.tasks.RecurringTask;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.states.StateName;
import cn.boboweike.carrot.utils.mapper.JsonMapper;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

public class CarrotApiHandler extends RestHttpHandler {
    private final PartitionedStorageProvider storageProvider;
    private final ProblemsManager problemsManager;
    private VersionUIModel versionUIModel;

    public CarrotApiHandler(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper) {
        super("/api", jsonMapper);
        this.storageProvider = storageProvider;
        this.problemsManager = new ProblemsManager(storageProvider);

        get("/tasks", findTaskByState());

        get("/tasks/:id", getTaskById());
        delete("/tasks/:id", deleteTaskById());
        post("/tasks/:id/requeue", requeueTaskById());

        get("/problems", getProblems());
        delete("/problems/:type", deleteProblemByType());

        get("/recurring-tasks", getRecurringTasks());
        delete("/recurring-tasks/:id", deleteRecurringTask());
        post("/recurring-tasks/:id/trigger", triggerRecurringTask());

        get("/servers", getBackgroundTaskServers());
        get("/partition-number", getNumberOfPartitions());
        get("/version", getVersion());

        withExceptionMapping(TaskNotFoundException.class, (exc, resp) -> resp.statusCode(404));
    }

    private HttpRequestHandler getTaskById() {
        return (request, response) -> {
            UUID id = request.param(":id", UUID.class);
            response.asJson(storageProvider.getTaskById(id));
        };
    }

    private HttpRequestHandler deleteTaskById() {
        return (request, response) -> {
            UUID id = request.param(":id", UUID.class);
            final Task task = storageProvider.getTaskById(id);
            task.delete("Task deleted via Dashboard");
            storageProvider.save(task);
            response.statusCode(204);
        };
    }

    private HttpRequestHandler requeueTaskById() {
        return (request, response) -> {
            UUID id = request.param(":id", UUID.class);
            final Task task = storageProvider.getTaskById(id);
            task.enqueue();
            storageProvider.save(task);
            response.statusCode(204);
        };
    }

    private HttpRequestHandler findTaskByState() {
        return (request, response) -> {
            Integer partition = request.queryParam("partition", Integer.class, 0);
            StateName stateName = request.queryParam("state", StateName.class, StateName.ENQUEUED);
            response.asJson(
                    storageProvider.getTaskPageByPartition(
                            stateName,
                            request.fromQueryParams(PageRequest.class),
                            partition
                    )
            );
        };
    }

    private HttpRequestHandler getProblems() {
        return (request, response) -> response.asJson(problemsManager.getProblems());
    }

    private HttpRequestHandler deleteProblemByType() {
        return (request, response) -> {
            String type = request.param(":type", String.class);
            problemsManager.dismissProblemOfType(type);
            response.statusCode(204);
        };
    }

    private HttpRequestHandler getRecurringTasks() {
        return (request, response) -> {
            final List<RecurringTaskUIModel> recurringTaskUIModels = storageProvider
                    .getRecurringTasks()
                    .stream()
                    .map(RecurringTaskUIModel::new)
                    .collect(Collectors.toList());
            response.asJson(recurringTaskUIModels);
        };
    }

    private HttpRequestHandler deleteRecurringTask() {
        return (request, response) -> {
            String id = request.param(":id");
            storageProvider.deleteRecurringTask(id);
            response.statusCode(204);
        };
    }

    private HttpRequestHandler triggerRecurringTask() {
        return (request, response) -> {
            String id = request.param(":id");
            final RecurringTask recurringTask = storageProvider.getRecurringTasks()
                    .stream()
                    .filter(rt -> id.equals(rt.getId()))
                    .findFirst()
                    .orElseThrow(() -> new TaskNotFoundException(id));

            final Task task = recurringTask.toEnqueuedTask();
            storageProvider.save(task);
            response.statusCode(204);
        };
    }

    private HttpRequestHandler getBackgroundTaskServers() {
        return (request, response) -> response.asJson(storageProvider.getBackgroundTaskServers());
    }

    private HttpRequestHandler getNumberOfPartitions() {
        NumberWrapper numberWrapper = new NumberWrapper();
        numberWrapper.number = storageProvider.getTotalNumOfPartitions();
        return (request, response) -> response.asJson(numberWrapper);
    }

    private HttpRequestHandler getVersion() {
        return (request, response) -> {
            response.asJson(getVersionUIModel());
        };
    }

    private VersionUIModel getVersionUIModel() {
        if (versionUIModel != null) return versionUIModel;
        this.versionUIModel = VersionUIModel.withoutAnonymousDataUsage();
        return this.versionUIModel;
    }

    static class NumberWrapper {
        public Integer number;
    }
}
