package cn.boboweike.carrot.dashboard;

import cn.boboweike.carrot.dashboard.server.HttpExchangeHandler;
import cn.boboweike.carrot.dashboard.server.WebServer;
import cn.boboweike.carrot.dashboard.server.http.RedirectHttpHandler;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.storage.ThreadSafePartitionedStorageProvider;
import cn.boboweike.carrot.utils.annotations.VisibleFor;
import cn.boboweike.carrot.utils.mapper.JsonMapper;
import cn.boboweike.carrot.utils.mapper.jackson.JacksonJsonMapper;
import com.sun.net.httpserver.BasicAuthenticator;
import com.sun.net.httpserver.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static cn.boboweike.carrot.dashboard.CarrotDashboardWebServerConfiguration.usingStandardDashboardConfiguration;
import static cn.boboweike.carrot.utils.StringUtils.isNotNullOrEmpty;

/**
 * Provides a dashboard which gives insights in your tasks and servers.
 * The dashboard server starts by default on port 8000.
 */
public class CarrotDashboardWebServer {
    private static final Logger LOGGER = LoggerFactory.getLogger(CarrotDashboardWebServer.class);

    private final PartitionedStorageProvider storageProvider;
    private final JsonMapper jsonMapper;
    private final int port;
    private final BasicAuthenticator basicAuthenticator;

    private WebServer webServer;

    public static void main(String[] args) {
        new CarrotDashboardWebServer(null, new JacksonJsonMapper());
    }

    public CarrotDashboardWebServer(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper) {
        this(storageProvider, jsonMapper, 8000);
    }

    public CarrotDashboardWebServer(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper, int port) {
        this(storageProvider, jsonMapper, usingStandardDashboardConfiguration().andPort(port));
    }

    public CarrotDashboardWebServer(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper, int port, String username, String password) {
        this(storageProvider, jsonMapper, usingStandardDashboardConfiguration().andPort(port).andBasicAuthentication(username, password));
    }

    public CarrotDashboardWebServer(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper, CarrotDashboardWebServerConfiguration configuration) {
        if (storageProvider == null)
            throw new IllegalArgumentException("A StorageProvider is required to use a CarrotDashboardWebServer. Please see the documentation on how to setup a task StorageProvider.");

        this.storageProvider = new ThreadSafePartitionedStorageProvider(storageProvider);
        this.jsonMapper = jsonMapper;
        this.port = configuration.port;
        this.basicAuthenticator = createOptionalBasicAuthenticator(configuration.username, configuration.password);
    }

    public void start() {
        RedirectHttpHandler redirectHttpHandler = new RedirectHttpHandler("/", "/dashboard");
        CarrotStaticFileHandler staticFileHandler = createStaticFileHandler();
        CarrotApiHandler dashboardHandler = createApiHandler(storageProvider, jsonMapper);
        CarrotSseHandler sseHandler = createSSeHandler(storageProvider, jsonMapper);

        webServer = new WebServer(port);
        registerContext(redirectHttpHandler);
        registerSecuredContext(staticFileHandler);
        registerSecuredContext(dashboardHandler);
        registerSecuredContext(sseHandler);
        webServer.start();

        LOGGER.info("Carrot Dashboard using {} started at http://{}:{}/dashboard",
                storageProvider.getName(),
                webServer.getWebServerHostAddress(),
                webServer.getWebServerHostPort());
    }

    public void stop() {
        if (webServer == null) return;
        webServer.stop();
        LOGGER.info("Carrot dashboard stopped");
        webServer = null;
    }

    HttpContext registerContext(HttpExchangeHandler httpHandler) {
        return webServer.createContext(httpHandler);
    }

    HttpContext registerSecuredContext(HttpExchangeHandler httpHandler) {
        HttpContext httpContext = registerContext(httpHandler);
        if (basicAuthenticator != null) {
            httpContext.setAuthenticator(basicAuthenticator);
        }
        return httpContext;
    }

    @VisibleFor("github issue 18")
    CarrotStaticFileHandler createStaticFileHandler() {
        return new CarrotStaticFileHandler();
    }

    @VisibleFor("github issue 18")
    CarrotApiHandler createApiHandler(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper) {
        return new CarrotApiHandler(storageProvider, jsonMapper);
    }

    @VisibleFor("github issue 18")
    CarrotSseHandler createSSeHandler(PartitionedStorageProvider storageProvider, JsonMapper jsonMapper) {
        return new CarrotSseHandler(storageProvider, jsonMapper);
    }

    private BasicAuthenticator createOptionalBasicAuthenticator(String username, String password) {
        if (isNotNullOrEmpty(username) && isNotNullOrEmpty(password)) {
            return new BasicAuthenticator("Carrot") {
                @Override
                public boolean checkCredentials(String user, String pwd) {
                    return user.equals(username) && pwd.equals(password);
                }
            };
        }
        return null;
    }
}
