package cn.boboweike.carrot.lock.inmemory;

import cn.boboweike.carrot.lock.LockProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;

/**
 * In memory lock that is suitable only for tests and running application locally.
 */
public class InMemoryLockProvider implements LockProvider {
    private final Map<String, LockRecord> locks = new HashMap<>();

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public boolean lock(String lockName, int durationInSeconds, String lockedBy) {
        synchronized (locks) {
            Instant now = Instant.now();
            if (isLocked(lockName, now)) {
                return false;
            } else {
                LockRecord lockRecord = new LockRecord(now.plusSeconds(durationInSeconds));
                locks.put(lockName, lockRecord);
                logger.debug("Locked {} with duration {}", lockName, durationInSeconds);
                return true;
            }
        }
    }

    boolean isLocked(String lockName, Instant now) {
        synchronized (locks) {
            LockRecord lockRecord = locks.get(lockName);
            return lockRecord != null && lockRecord.lockedUntil.isAfter(now);
        }
    }

    @Override
    public boolean extend(String lockName, int durationInSeconds, String lockedBy) {
        synchronized (locks) {
            Instant now = Instant.now();
            if (isLocked(lockName, now)) {
                locks.put(lockName, new LockRecord(now.plusSeconds(durationInSeconds)));
                logger.debug("Extended {} with duration {}", lockName, durationInSeconds);
                return true;
            } else {
                return false;
            }
        }
    }

    @Override
    public boolean unlock(String lockName) {
        synchronized (locks) {
            Instant now = Instant.now();
            locks.put(lockName, new LockRecord(now));
        }
        return true;
    }

    private static class LockRecord {
        private final Instant lockedUntil;

        private LockRecord(Instant lockedUntil) {
            this.lockedUntil = lockedUntil;
        }
    }
}
