package cn.boboweike.carrot.lock.nosql;

import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.FindOneAndUpdateOptions;
import cn.boboweike.carrot.lock.LockProvider;
import org.bson.Document;
import org.bson.conversions.Bson;

import java.time.Instant;

import static com.mongodb.client.model.Filters.*;
import static com.mongodb.client.model.Updates.combine;
import static com.mongodb.client.model.Updates.set;


/**
 * Adapted from https://github.com/lukas-krecan/ShedLock/blob/master/providers/mongo/shedlock-provider-mongo/src/main/java/net/javacrumbs/shedlock/provider/mongo/MongoLockProvider.java
 *
 * Distributed lock using MongoDB &gt;= 2.6. Requires mongo-java-driver &gt; 3.4.0
 * <p>
 * It uses a collection that contains documents like this:
 * <pre>
 * {
 *    "_id" : "lock name",
 *    "lockUntil" : ISODate("2017-01-07T16:52:04.071Z"),
 *    "lockedAt" : ISODate("2017-01-07T16:52:03.932Z"),
 *    "lockedBy" : "host name"
 * }
 * </pre>
 *
 * lockedAt and lockedBy are just for troubleshooting and are not read by the code
 *
 * <ol>
 * <li>
 * Attempts to insert a new lock record. As an optimization, we keep in-memory track of created lock records. If the record
 * has been inserted, returns lock.
 * </li>
 * <li>
 * We will try to update lock record using filter _id == name AND lock_until &lt;= now
 * </li>
 * <li>
 * If the update succeeded (1 updated document), we have the lock. If the update failed (0 updated documents) somebody else holds the lock
 * </li>
 * <li>
 * When unlocking, lock_until is set to now.
 * </li>
 * </ol>
 */
public class MongoLockProvider implements LockProvider {

    static final String ID = "_id";

    private final MongoCollection<Document> collection;

    /**
     * Uses Mongo to coordinate locks
     *
     * @param collection Mongo collection to be used
     */
    public MongoLockProvider(MongoCollection<Document> collection) {
        this.collection = collection;
    }


    @Override
    public boolean lock(String name, int durationInSeconds, String lockedBy) {
        Instant now = Instant.now();
        Bson update = combine(
                set(LOCK_UNTIL, now.plusSeconds(durationInSeconds)),
                set(LOCKED_AT, now),
                set(LOCKED_BY, lockedBy)
        );
        try {
            // There are three possible situations:
            // 1. The lock document does not exist yet - it is inserted - we have the lock
            // 2. The lock document exists and lockUtil <= now - it is updated - we have the lock
            // 3. The lock document exists and lockUtil > now - Duplicate key exception is thrown
            this.collection.findOneAndUpdate(
                    and(eq(ID, name), lte(LOCK_UNTIL, now)),
                    update,
                    new FindOneAndUpdateOptions().upsert(true)
            );
            return true;
        } catch (Exception e) {
//            if (e.getCode() == 11000) { // duplicate key
//                //Upsert attempts to insert when there were no filter matches.
//                //This means there was a lock with matching ID with lockUntil > now.
//                return false;
//            } else {
//                throw e;
//            }
            return false;
        }
    }

    @Override
    public boolean extend(String name, int durationInSeconds, String lockedBy) {
        Instant now = Instant.now();
        Bson update = set(LOCK_UNTIL, now.plusSeconds(durationInSeconds));

        try {
            Document updatedDocument = collection.findOneAndUpdate(
                    and(
                            eq(ID, name),
                            gt(LOCK_UNTIL, now),
                            eq(LOCKED_BY, lockedBy)
                    ),
                    update
            );
            return updatedDocument != null;
        } catch (Exception ex) {
            return false;
        }
    }

    @Override
    public boolean unlock(String name) {
        // Set lockUtil to now
        try {
            collection.findOneAndUpdate(
                    eq(ID, name),
                    combine(set(LOCK_UNTIL, Instant.now()))
            );
            return true;
        } catch (Exception ex) {
            return false;
        }
    }
}
