package cn.boboweike.carrot.scheduling;

import cn.boboweike.carrot.configuration.Carrot;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.RecurringTask;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.filters.TaskDefaultFilters;
import cn.boboweike.carrot.tasks.filters.TaskFilter;
import cn.boboweike.carrot.tasks.filters.TaskFilterUtils;
import cn.boboweike.carrot.tasks.mappers.MDCMapper;
import cn.boboweike.carrot.tasks.states.ScheduledState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.time.ZoneId;
import java.util.List;
import java.util.UUID;

import static java.util.Collections.emptyList;

public class AbstractTaskScheduler {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractTaskScheduler.class);

    private final PartitionedStorageProvider storageProvider;
    private final TaskFilterUtils taskFilterUtils;

    /**
     * Creates a new AbstractTaskScheduler using the provided storageProvider
     *
     * @param storageProvider the storageProvider to use
     */
    public AbstractTaskScheduler(PartitionedStorageProvider storageProvider) {
        this(storageProvider, emptyList());
    }

    /**
     * Creates a new AbstractTaskScheduler using the provided storageProvider and the list of TaskFilters that will be used for every background task
     *
     * @param storageProvider the storageProvider to use
     * @param taskFilters      list of taskFilters that will be used for every task
     */
    public AbstractTaskScheduler(PartitionedStorageProvider storageProvider, List<TaskFilter> taskFilters) {
        if (storageProvider == null)
            throw new IllegalArgumentException("A TaskStorageProvider is required to use the TaskScheduler. Please see the documentation on how to setup a TaskStorageProvider.");
        this.storageProvider = storageProvider;
        this.taskFilterUtils = new TaskFilterUtils(new TaskDefaultFilters(taskFilters));
    }

    /**
     * @see #delete(UUID)
     * @param taskId the id of the task
     */
    public void delete(TaskId taskId) {
        this.delete(taskId.asUUID());
    }

    /**
     * @see #delete(UUID, String)
     * @param taskId the id of the task
     * @param reason the reason to delete
     */
    public void delete(TaskId taskId, String reason) {
        this.delete(taskId.asUUID(), reason);
    }

    /**
     * Deletes a task and sets its state to DELETED. If the task is being processed, it will be interrupted.
     *
     * @param id the id of the task
     */
    public void delete(UUID id) {
        delete(id, "Deleted via TaskScheduler API");
    }

    /**
     * Deletes a task and sets its state to DELETED. If the task is being processed, it will be interrupted.
     *
     * @param id     the id of the task
     * @param reason the reason why the task is deleted.
     */
    public void delete(UUID id, String reason) {
        final Task taskToDelete = storageProvider.getTaskById(id);
        final Task task = storageProvider.getTaskById(id);
        taskToDelete.delete(reason);
        taskFilterUtils.runOnStateElectionFilter(taskToDelete);
        final Task deletedTask = storageProvider.save(taskToDelete);
        taskFilterUtils.runOnStateAppliedFilters(deletedTask);
        LOGGER.debug("Deleted Task with id {}", deletedTask.getId());
    }

    /**
     * Deletes the recurring task based on the given id.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.delete("my-recurring-task"));
     * }</pre>
     *
     * @param id the id of the recurring task to delete
     */
    public void delete(String id) {
        this.storageProvider.deleteRecurringTask(id);
    }

    /**
     * Utility method to register the shutdown of Carrot in various containers - it is even automatically called by Spring Framework.
     * Note that this will stop the BackgroundTaskServer, the Dashboard and the StorageProvider. TaskProcessing will stop and enqueueing new tasks will fail.
     */
    public void shutdown() {
        Carrot.destroy();
    }

    TaskId enqueue(UUID id, TaskDetails taskDetails) {
        return saveTask(new Task(id, taskDetails));
    }

    TaskId schedule(UUID id, Instant scheduleAt, TaskDetails taskDetails) {
        return saveTask(new Task(id, taskDetails, new ScheduledState(scheduleAt)));
    }

    String scheduleRecurrently(String id, TaskDetails taskDetails, Schedule schedule, ZoneId zoneId) {
        final RecurringTask recurringTask = new RecurringTask(id, taskDetails, schedule, zoneId);
        taskFilterUtils.runOnCreatingFilter(recurringTask);
        RecurringTask savedRecurringTask = this.storageProvider.saveRecurringTask(recurringTask);
        taskFilterUtils.runOnCreatedFilter(recurringTask);
        return savedRecurringTask.getId();
    }

    TaskId saveTask(Task task) {
        try {
            MDCMapper.saveMDCContextToTask(task);
            taskFilterUtils.runOnCreatingFilter(task);
            Task savedTask = this.storageProvider.save(task);
            taskFilterUtils.runOnCreatedFilter(savedTask);
            LOGGER.debug("Created Task with id {}", task.getId());
        } catch (ConcurrentTaskModificationException e) {
            LOGGER.info("Skipped Task with id {} as it already exists", task.getId());
        }
        return new TaskId(task.getId());
    }

    List<Task> saveTasks(List<Task> tasks) {
        tasks.forEach(MDCMapper::saveMDCContextToTask);
        taskFilterUtils.runOnCreatingFilter(tasks);
        final List<Task> savedTasks = this.storageProvider.save(tasks);
        taskFilterUtils.runOnCreatedFilter(savedTasks);
        return savedTasks;
    }
}
