package cn.boboweike.carrot.scheduling;

import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.lambdas.IocTaskLambda;
import cn.boboweike.carrot.tasks.lambdas.IocTaskLambdaFromStream;
import cn.boboweike.carrot.tasks.lambdas.TaskLambda;
import cn.boboweike.carrot.tasks.lambdas.TaskLambdaFromStream;

import java.time.*;
import java.util.UUID;
import java.util.stream.Stream;

/**
 * Provides static methods for creating fire-and-forget, delayed and recurring tasks as well as to delete existing background tasks.
 * If you prefer not to use a static accessor, you can inject the {@link TaskScheduler} which exposes the same methods.
 *
 * @author Ronald Dehuysser
 */
public class BackgroundTask {

    BackgroundTask() {
    }

    private static TaskScheduler taskScheduler;

    /**
     * Creates a new fire-and-forget task based on a given lambda.
     * <h3>An example:</h3>
     * <pre>{@code
     *            MyService service = new MyService();
     *            BackgroundTask.enqueue(() -> service.doWork());
     *       }</pre>
     *
     * @param task the lambda which defines the fire-and-forget task
     * @return the id of the task
     */
    public static TaskId enqueue(TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.enqueue(task);
    }

    /**
     * Creates a new fire-and-forget task based on a given lambda.
     * If a task with that id already exists, Carrot will not save it again.
     *
     * <h3>An example:</h3>
     * <pre>{@code
     *            MyService service = new MyService();
     *            BackgroundTask.enqueue(id, () -> service.doWork());
     *       }</pre>
     *
     * @param id  the uuid with which to save the task
     * @param task the lambda which defines the fire-and-forget task
     * @return the id of the task
     */
    public static TaskId enqueue(UUID id, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.enqueue(id, task);
    }

    /**
     * Creates new fire-and-forget tasks for each item in the input stream using the lambda passed as {@code taskFromStream}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      Stream<UUID> workStream = getWorkStream();
     *      BackgroundTask.enqueue(workStream, (uuid) -> service.doWork(uuid));
     * }</pre>
     *
     * @param input          the stream of items for which to create fire-and-forget tasks
     * @param taskFromStream the lambda which defines the fire-and-forget task to create for each item in the {@code input}
     * @param <T>            generic type for the Stream
     */
    public static <T> void enqueue(Stream<T> input, TaskLambdaFromStream<T> taskFromStream) {
        verifyTaskScheduler();
        taskScheduler.enqueue(input, taskFromStream);
    }

    /**
     * Creates a new fire-and-forget task based on a given lambda. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *            BackgroundTask.<MyService>enqueue(x -> x.doWork());
     *       }</pre>
     *
     * @param iocTask the lambda which defines the fire-and-forget task
     * @return the id of the task
     * @param <S> generic type for the IocTaskLambda
     */
    public static <S> TaskId enqueue(IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.enqueue(iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on a given lambda. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *            BackgroundTask.<MyService>enqueue(id, x -> x.doWork());
     *       }</pre>
     *
     * @param id     the uuid with which to save the task
     * @param iocTask the lambda which defines the fire-and-forget task
     * @param <S> generic type for the IocTaskLambda
     * @return the id of the task
     */
    public static <S> TaskId enqueue(UUID id, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.enqueue(id, iocTask);
    }

    /**
     * Creates new fire-and-forget tasks for each item in the input stream using the lambda passed as {@code taskFromStream}. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      Stream<UUID> workStream = getWorkStream();
     *      BackgroundTask.<MyService, UUID>enqueue(workStream, (x, uuid) -> x.doWork(uuid));
     * }</pre>
     *
     * @param input            the stream of items for which to create fire-and-forget tasks
     * @param iocTaskFromStream the lambda which defines the fire-and-forget task to create for each item in the {@code input}
     * @param <S> generic type for the IocTaskLambdaFromStream
     * @param <T> generic type for the IocTaskLambdaFromStream
     */
    public static <S, T> void enqueue(Stream<T> input, IocTaskLambdaFromStream<S, T> iocTaskFromStream) {
        verifyTaskScheduler();
        taskScheduler.enqueue(input, iocTaskFromStream);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(ZonedDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param zonedDateTime The moment in time at which the task will be enqueued.
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(ZonedDateTime zonedDateTime, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(zonedDateTime, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(id, ZonedDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param zonedDateTime The moment in time at which the task will be enqueued.
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, ZonedDateTime zonedDateTime, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, zonedDateTime, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(ZonedDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param zonedDateTime The moment in time at which the task will be enqueued.
     * @param iocTask        the lambda which defines the fire-and-forget task
     * @param <S> generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(ZonedDateTime zonedDateTime, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(zonedDateTime, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(id, ZonedDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param zonedDateTime The moment in time at which the task will be enqueued.
     * @param iocTask        the lambda which defines the fire-and-forget task\
     * @param <S> generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(UUID id, ZonedDateTime zonedDateTime, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, zonedDateTime, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(OffsetDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param task            the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(OffsetDateTime offsetDateTime, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(offsetDateTime, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(id, OffsetDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id             the uuid with which to save the task
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param task            the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, OffsetDateTime offsetDateTime, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, offsetDateTime, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(id, OffsetDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param iocTask         the lambda which defines the fire-and-forget task
     * @param <S> generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(OffsetDateTime offsetDateTime, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(offsetDateTime, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(id, OffsetDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id             the uuid with which to save the task
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param iocTask         the lambda which defines the fire-and-forget task
     * @param <S>            generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(UUID id, OffsetDateTime offsetDateTime, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, offsetDateTime, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(LocalDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param localDateTime The moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(LocalDateTime localDateTime, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(localDateTime, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(id, LocalDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param localDateTime The moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, LocalDateTime localDateTime, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, localDateTime, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(LocalDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param localDateTime The moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param iocTask       the lambda which defines the fire-and-forget task
     * @param <S>           generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(LocalDateTime localDateTime, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(localDateTime, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(id, LocalDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param localDateTime The moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param iocTask        the lambda which defines the fire-and-forget task
     * @param <S>           generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(UUID id, LocalDateTime localDateTime, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, localDateTime, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(Instant.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param instant The moment in time at which the task will be enqueued.
     * @param task     the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(Instant instant, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(instant, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.schedule(id, Instant.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id      the uuid with which to save the task
     * @param instant The moment in time at which the task will be enqueued.
     * @param task     the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, Instant instant, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, instant, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     *
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(Instant.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param instant The moment in time at which the task will be enqueued.
     * @param iocTask  the lambda which defines the fire-and-forget task
     * @param <S>     Generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(Instant instant, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(instant, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>schedule(id, Instant.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id      the uuid with which to save the task
     * @param instant The moment in time at which the task will be enqueued.
     * @param iocTask  the lambda which defines the fire-and-forget task
     * @param <S>     Generic type for the IocTaskLambda
     * @return the id of the Task
     */
    public static <S> TaskId schedule(UUID id, Instant instant, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.schedule(id, instant, iocTask);
    }

    /**
     * Deletes a task and sets its state to DELETED. If the task is being processed, it will be interrupted.
     *
     * @param id the id of the task
     */
    public static void delete(UUID id) {
        verifyTaskScheduler();
        taskScheduler.delete(id);
    }

    /**
     * @see #delete(UUID)
     *
     * @param taskId the id of the task
     */
    public static void delete(TaskId taskId) {
        delete(taskId.asUUID());
    }

    /**
     * Creates a new recurring task based on the given cron expression and the given lambda. The tasks will be scheduled using the systemDefault timezone.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently(Cron.daily(), () -> service.doWork());
     * }</pre>
     *
     * @param cron The cron expression defining when to run this recurring task
     * @param task  the lambda which defines the fire-and-forget task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static String scheduleRecurrently(String cron, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(cron, task);
    }

    /**
     * Creates a new recurring task based on the given cron expression and the given lambda. The IoC container will be used to resolve {@code MyService}. The tasks will be scheduled using the systemDefault timezone.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently(Cron.daily(), x -> x.doWork());
     * }</pre>
     *
     * @param cron   The cron expression defining when to run this recurring task
     * @param iocTask the lambda which defines the fire-and-forget task
     * @param <S>    Generic type for the IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static <S> String scheduleRecurrently(String cron, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(cron, iocTask);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression and lambda. The tasks will be scheduled using the systemDefault timezone
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently("my-recurring-task", Cron.daily(), () -> service.doWork());
     * }</pre>
     *
     * @param id   the id of this recurring task which can be used to alter or delete it
     * @param cron The cron expression defining when to run this recurring task
     * @param task  the lambda which defines the fire-and-forget task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static String scheduleRecurrently(String id, String cron, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(id, cron, task);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression and lambda. The IoC container will be used to resolve {@code MyService}. The tasks will be scheduled using the systemDefault timezone
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently("my-recurring-task", Cron.daily(), x -> x.doWork());
     * }</pre>
     *
     * @param id     the id of this recurring task which can be used to alter or delete it
     * @param cron   The cron expression defining when to run this recurring task
     * @param iocTask the lambda which defines the fire-and-forget task
     * @param <S>    Generic type for the IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static <S> String scheduleRecurrently(String id, String cron, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(id, cron, iocTask);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression, {@code ZoneId} and lambda.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently("my-recurring-task", Cron.daily(), ZoneId.of("Europe/Brussels"), () -> service.doWork());
     * }</pre>
     *
     * @param id     the id of this recurring task which can be used to alter or delete it
     * @param cron   The cron expression defining when to run this recurring task
     * @param zoneId The zoneId (timezone) of when to run this recurring task
     * @param task    the lambda which defines the fire-and-forget task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static String scheduleRecurrently(String id, String cron, ZoneId zoneId, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(id, cron, zoneId, task);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression, {@code ZoneId} and lambda. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently("my-recurring-task", Cron.daily(), ZoneId.of("Europe/Brussels"), x -> x.doWork());
     * }</pre>
     *
     * @param id     the id of this recurring task which can be used to alter or delete it
     * @param cron   The cron expression defining when to run this recurring task
     * @param zoneId The zoneId (timezone) of when to run this recurring task
     * @param iocTask the lambda which defines the fire-and-forget task
     * @param <S>    Generic type for the IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static <S> String scheduleRecurrently(String id, String cron, ZoneId zoneId, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(id, cron, zoneId, iocTask);
    }

    /**
     * Creates a new recurring task based on the given duration and the given lambda. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently(Duration.parse("P5D"), () -> service.doWork());
     * }</pre>
     *
     * @param duration the duration defining the time between each instance of this recurring task
     * @param task      the lambda which defines the fire-and-forget task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public static String scheduleRecurrently(Duration duration, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(duration, task);
    }

    /**
     * Creates a new recurring task based on the given duration and the given lambda. The IoC container will be used to resolve {@code MyService}. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently(Duration.parse("P5D"), x -> x.doWork());
     * }</pre>
     *
     * @param duration the duration defining the time between each instance of this recurring task
     * @param iocTask   the lambda which defines the fire-and-forget task
     * @param <S>      Generic type for the IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public static <S> String scheduleRecurrently(Duration duration, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(duration, iocTask);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, duration and lambda. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently("my-recurring-task", Duration.parse("P5D"), () -> service.doWork());
     * }</pre>
     *
     * @param id       the id of this recurring task which can be used to alter or delete it
     * @param duration the duration defining the time between each instance of this recurring task
     * @param task      the lambda which defines the fire-and-forget task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public static String scheduleRecurrently(String id, Duration duration, TaskLambda task) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(id, duration, task);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, duration and lambda. The IoC container will be used to resolve {@code MyService}. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently("my-recurring-task", Duration.parse("P5D"), x -> x.doWork());
     * }</pre>
     *
     * @param id       the id of this recurring task which can be used to alter or delete it
     * @param duration the duration defining the time between each instance of this recurring task
     * @param iocTask   the lambda which defines the fire-and-forget task
     * @param <S>      generic type for the IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public static <S> String scheduleRecurrently(String id, Duration duration, IocTaskLambda<S> iocTask) {
        verifyTaskScheduler();
        return taskScheduler.scheduleRecurrently(id, duration, iocTask);
    }

    /**
     * Deletes the recurring task based on the given id.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.delete("my-recurring-task"));
     * }</pre>
     *
     * @param id the id of the recurring task to delete
     */
    public static void delete(String id) {
        verifyTaskScheduler();
        taskScheduler.delete(id);
    }

    private static void verifyTaskScheduler() {
        if (taskScheduler != null) return;
        throw new IllegalStateException("The TaskScheduler has not been initialized. Use the fluent Carrot.configure() API to setup Carrot or set the TaskScheduler via the static setter method.");
    }

    public static void setTaskScheduler(TaskScheduler taskScheduler) {
        BackgroundTask.taskScheduler = taskScheduler;
    }
}
