package cn.boboweike.carrot.scheduling;

import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.lambdas.TaskRequest;

import java.time.*;
import java.util.UUID;
import java.util.stream.Stream;

import static java.time.ZoneId.systemDefault;

/**
 * Provides static methods for creating fire-and-forget, delayed and recurring tasks as well as to delete existing background tasks.
 * If you prefer not to use a static accessor, you can inject the {@link TaskRequestScheduler} which exposes the same methods.
 *
 * @author Ronald Dehuysser
 */
public class BackgroundTaskRequest {

    private BackgroundTaskRequest() {
    }

    private static TaskRequestScheduler taskRequestScheduler;

    /**
     * Creates a new fire-and-forget task based on a given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *            BackgroundTaskRequest.enqueue(new MyTaskRequest());
     *       }</pre>
     *
     * @param taskRequest the taskRequest which defines the fire-and-forget task.
     * @return the id of the task
     */
    public static TaskId enqueue(TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.enqueue(taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on a given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *            BackgroundTaskRequest.enqueueTaskRequest(id, new MyTaskRequest());
     *       }</pre>
     *
     * @param id         the uuid with which to save the task
     * @param taskRequest the taskRequest which defines the fire-and-forget task.
     * @return the id of the task
     */
    public static TaskId enqueue(UUID id, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.enqueue(id, taskRequest);
    }

    /**
     * Creates new fire-and-forget tasks for each item in the input stream. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      Stream<MyTaskRequest> workStream = getWorkStream();
     *      BackgroundTaskRequest.enqueue(workStream);
     * }</pre>
     *
     * @param input the stream of taskRequests for which to create fire-and-forget tasks
     */
    public static void enqueue(Stream<? extends TaskRequest> input) {
        verifyTaskScheduler();
        taskRequestScheduler.enqueue(input);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(ZonedDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(ZonedDateTime zonedDateTime, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(zonedDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(id, ZonedDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, ZonedDateTime zonedDateTime, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(id, zonedDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(OffsetDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param offsetDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest     the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(OffsetDateTime offsetDateTime, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(offsetDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(id, OffsetDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id             the uuid with which to save the task
     * @param offsetDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest     the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, OffsetDateTime offsetDateTime, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(id, offsetDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(LocalDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(LocalDateTime localDateTime, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(localDateTime.atZone(systemDefault()).toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(id, LocalDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, LocalDateTime localDateTime, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(id, localDateTime.atZone(systemDefault()).toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(Instant.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param instant    the moment in time at which the task will be enqueued.
     * @param taskRequest the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(Instant instant, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(instant, taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.schedule(id, Instant.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id         the uuid with which to save the task
     * @param instant    the moment in time at which the task will be enqueued.
     * @param taskRequest the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public static TaskId schedule(UUID id, Instant instant, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.schedule(id, instant, taskRequest);
    }

    /**
     * Deletes a task and sets its state to DELETED. If the task is being processed, it will be interrupted.
     *
     * @param id the id of the task
     */
    public static void delete(UUID id) {
        verifyTaskScheduler();
        taskRequestScheduler.delete(id);
    }

    /**
     * @see #delete(UUID)
     * @param taskId the id of the task
     */
    public static void delete(TaskId taskId) {
        delete(taskId.asUUID());
    }

    /**
     * Creates a new recurring task based on the given cron expression and the given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor. The tasks will be scheduled using the systemDefault timezone.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.scheduleRecurrently(Cron.daily(), new MyTaskRequest());
     * }</pre>
     *
     * @param cron       The cron expression defining when to run this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static String scheduleRecurrently(String cron, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.scheduleRecurrently(cron, taskRequest);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression and taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor. The tasks will be scheduled using the systemDefault timezone
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.scheduleRecurrently("my-recurring-task", Cron.daily(), new MyTaskRequest());
     * }</pre>
     *
     * @param id         the id of this recurring task which can be used to alter or delete it
     * @param cron       The cron expression defining when to run this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static String scheduleRecurrently(String id, String cron, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.scheduleRecurrently(id, cron, systemDefault(), taskRequest);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression, {@code ZoneId} and taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.scheduleRecurrently("my-recurring-task", Cron.daily(), ZoneId.of("Europe/Brussels"), new MyTaskRequest());
     * }</pre>
     *
     * @param id         the id of this recurring task which can be used to alter or delete it
     * @param cron       The cron expression defining when to run this recurring task
     * @param zoneId     The zoneId (timezone) of when to run this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public static String scheduleRecurrently(String id, String cron, ZoneId zoneId, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.scheduleRecurrently(id, cron, zoneId, taskRequest);
    }

    /**
     * Creates a new recurring task based on the given duration and the given taskRequest. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently(Duration.parse("P5D"), () -> service.doWork());
     * }</pre>
     *
     * @param duration the duration defining the time between each instance of this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public static String scheduleRecurrently(Duration duration, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.scheduleRecurrently(duration, taskRequest);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, duration and taskRequest. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently("my-recurring-task", Duration.parse("P5D"), () -> service.doWork());
     * }</pre>
     *
     * @param id       the id of this recurring task which can be used to alter or delete it
     * @param duration the duration defining the time between each instance of this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public static String scheduleRecurrently(String id, Duration duration, TaskRequest taskRequest) {
        verifyTaskScheduler();
        return taskRequestScheduler.scheduleRecurrently(id, duration, taskRequest);
    }

    /**
     * Deletes the recurring task based on the given id.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTaskRequest.delete("my-recurring-task"));
     * }</pre>
     *
     * @param id the id of the recurring task to delete
     */
    public static void delete(String id) {
        verifyTaskScheduler();
        taskRequestScheduler.delete(id);
    }

    private static void verifyTaskScheduler() {
        if (taskRequestScheduler != null) return;
        throw new IllegalStateException("The TaskRequestScheduler has not been initialized. Use the fluent Carrot.configure() API to setup Carrot or set the TaskRequestScheduler via the static setter method.");
    }

    public static void setTaskRequestScheduler(TaskRequestScheduler taskRequestScheduler) {
        BackgroundTaskRequest.taskRequestScheduler = taskRequestScheduler;
    }
}

