package cn.boboweike.carrot.scheduling;

import cn.boboweike.carrot.scheduling.cron.CronExpression;
import cn.boboweike.carrot.scheduling.interval.Interval;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.filters.TaskFilter;
import cn.boboweike.carrot.tasks.lambdas.TaskRequest;

import java.time.*;
import java.util.List;
import java.util.UUID;
import java.util.stream.Stream;

import static cn.boboweike.carrot.storage.PartitionedStorageProvider.BATCH_SIZE;
import static cn.boboweike.carrot.utils.streams.StreamUtils.batchCollector;
import static java.time.ZoneId.systemDefault;
import static java.util.Collections.emptyList;

/**
 * Provides methods for creating fire-and-forget, delayed and recurring tasks as well as to delete existing background tasks.
 * <p>
 * This {@code TaskRequestScheduler} allows to schedule tasks by means of an implementation of a {@code TaskRequest}.
 *
 * @author Ronald Dehuysser
 */
public class TaskRequestScheduler extends AbstractTaskScheduler {

    /**
     * Creates a new TaskRequestScheduler using the provided storageProvider
     *
     * @param storageProvider the storageProvider to use
     */
    public TaskRequestScheduler(PartitionedStorageProvider storageProvider) {
        super(storageProvider, emptyList());
    }

    /**
     * Creates a new TaskRequestScheduler using the provided storageProvider and the list of TaskFilters that will be used for every background task
     *
     * @param storageProvider the storageProvider to use
     * @param taskFilters      list of taskFilters that will be used for every task
     */
    public TaskRequestScheduler(PartitionedStorageProvider storageProvider, List<TaskFilter> taskFilters) {
        super(storageProvider, taskFilters);
        BackgroundTaskRequest.setTaskRequestScheduler(this);
    }

    /**
     * Creates a new fire-and-forget task based on a given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *            taskScheduler.enqueue(new MyTaskRequest());
     *       }</pre>
     *
     * @param taskRequest the taskRequest which defines the fire-and-forget task.
     * @return the id of the task
     */
    public TaskId enqueue(TaskRequest taskRequest) {
        return enqueue(null, taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on a given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *            taskScheduler.enqueue(id, new MyTaskRequest());
     *       }</pre>
     *
     * @param id         the uuid with which to save the task
     * @param taskRequest the taskRequest which defines the fire-and-forget task.
     * @return the id of the task
     */
    public TaskId enqueue(UUID id, TaskRequest taskRequest) {
        TaskDetails taskDetails = new TaskDetails(taskRequest);
        return enqueue(id, taskDetails);
    }

    /**
     * Creates new fire-and-forget tasks for each item in the input stream. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      Stream<MyTaskRequest> workStream = getWorkStream();
     *      taskScheduler.enqueue(workStream);
     * }</pre>
     *
     * @param input the stream of taskRequests for which to create fire-and-forget tasks
     */
    public void enqueue(Stream<? extends TaskRequest> input) {
        input
                .map(TaskDetails::new)
                .map(Task::new)
                .collect(batchCollector(BATCH_SIZE, this::saveTasks));
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(ZonedDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(ZonedDateTime zonedDateTime, TaskRequest taskRequest) {
        return schedule(null, zonedDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(id, ZonedDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, ZonedDateTime zonedDateTime, TaskRequest taskRequest) {
        return schedule(id, zonedDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(OffsetDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param offsetDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest     the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(OffsetDateTime offsetDateTime, TaskRequest taskRequest) {
        return schedule(null, offsetDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(id, OffsetDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id             the uuid with which to save the task
     * @param offsetDateTime the moment in time at which the task will be enqueued.
     * @param taskRequest     the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, OffsetDateTime offsetDateTime, TaskRequest taskRequest) {
        return schedule(id, offsetDateTime.toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(LocalDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(LocalDateTime localDateTime, TaskRequest taskRequest) {
        return schedule(localDateTime.atZone(systemDefault()).toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(id, LocalDateTime.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param taskRequest    the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, LocalDateTime localDateTime, TaskRequest taskRequest) {
        return schedule(id, localDateTime.atZone(systemDefault()).toInstant(), taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(Instant.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param instant    the moment in time at which the task will be enqueued.
     * @param taskRequest the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(Instant instant, TaskRequest taskRequest) {
        return schedule(null, instant, taskRequest);
    }

    /**
     * Creates a new fire-and-forget task based on the given taskRequest and schedules it to be enqueued at the given moment of time. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.schedule(id, Instant.now().plusHours(5), new MyTaskRequest());
     * }</pre>
     *
     * @param id         the uuid with which to save the task
     * @param instant    the moment in time at which the task will be enqueued.
     * @param taskRequest the taskRequest which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, Instant instant, TaskRequest taskRequest) {
        TaskDetails taskDetails = new TaskDetails(taskRequest);
        return schedule(id, instant, taskDetails);
    }

    /**
     * Creates a new recurring task based on the given cron expression and the given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor. The tasks will be scheduled using the systemDefault timezone.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.scheduleRecurrently(Cron.daily(), new MyTaskRequest());
     * }</pre>
     *
     * @param cron       The cron expression defining when to run this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public String scheduleRecurrently(String cron, TaskRequest taskRequest) {
        return scheduleRecurrently(null, cron, taskRequest);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression and taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor. The tasks will be scheduled using the systemDefault timezone
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.scheduleRecurrently("my-recurring-task", Cron.daily(), new MyTaskRequest());
     * }</pre>
     *
     * @param id         the id of this recurring task which can be used to alter or delete it
     * @param cron       The cron expression defining when to run this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public String scheduleRecurrently(String id, String cron, TaskRequest taskRequest) {
        return scheduleRecurrently(id, cron, systemDefault(), taskRequest);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression, {@code ZoneId} and taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.scheduleRecurrently("my-recurring-task", Cron.daily(), ZoneId.of("Europe/Brussels"), new MyTaskRequest());
     * }</pre>
     *
     * @param id         the id of this recurring task which can be used to alter or delete it
     * @param cron       The cron expression defining when to run this recurring task
     * @param zoneId     The zoneId (timezone) of when to run this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public String scheduleRecurrently(String id, String cron, ZoneId zoneId, TaskRequest taskRequest) {
        TaskDetails taskDetails = new TaskDetails(taskRequest);
        return scheduleRecurrently(id, taskDetails, CronExpression.create(cron), zoneId);
    }

    /**
     * Creates a new recurring task based on the given duration and the given taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor. The first run of this recurring task will happen
     * after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently(Duration.parse("P5D"), new MyTaskRequest());
     * }</pre>
     *
     * @param duration the duration defining the time between each instance of this recurring task.
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public String scheduleRecurrently(Duration duration, TaskRequest taskRequest) {
        return scheduleRecurrently(null, duration, taskRequest);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, duration and taskRequest. Carrot will try to find the TaskRequestHandler in
     * the IoC container or else it will try to create the handler by calling the default no-arg constructor. The first run of this recurring task will happen
     * after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently("my-recurring-task", Duration.parse("P5D"), new MyTaskRequest());
     * }</pre>
     *
     * @param id       the id of this recurring task which can be used to alter or delete it
     * @param duration the duration defining the time between each instance of this recurring task
     * @param taskRequest the taskRequest which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public String scheduleRecurrently(String id, Duration duration, TaskRequest taskRequest) {
        TaskDetails taskDetails = new TaskDetails(taskRequest);
        return scheduleRecurrently(id, taskDetails, new Interval(duration), systemDefault());
    }
}

