package cn.boboweike.carrot.scheduling;

import cn.boboweike.carrot.scheduling.cron.CronExpression;
import cn.boboweike.carrot.scheduling.interval.Interval;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.tasks.details.TaskDetailsAsmGenerator;
import cn.boboweike.carrot.tasks.details.TaskDetailsGenerator;
import cn.boboweike.carrot.tasks.filters.TaskFilter;
import cn.boboweike.carrot.tasks.lambdas.IocTaskLambda;
import cn.boboweike.carrot.tasks.lambdas.IocTaskLambdaFromStream;
import cn.boboweike.carrot.tasks.lambdas.TaskLambda;
import cn.boboweike.carrot.tasks.lambdas.TaskLambdaFromStream;

import java.time.*;
import java.util.List;
import java.util.UUID;
import java.util.stream.Stream;

import static cn.boboweike.carrot.storage.PartitionedStorageProvider.BATCH_SIZE;
import static cn.boboweike.carrot.utils.streams.StreamUtils.batchCollector;
import static java.time.ZoneId.systemDefault;
import static java.util.Collections.emptyList;

/**
 * Provides methods for creating fire-and-forget, delayed and recurring tasks as well as to delete existing background tasks.
 * <p>
 * This {@code TaskScheduler} allows to schedule tasks by means of a Java 8 lambda which is analyzed.
 *
 * @author boboweike
 */
public class TaskScheduler extends AbstractTaskScheduler {

    private final TaskDetailsGenerator taskDetailsGenerator;

    /**
     * Creates a new TaskScheduler using the provided storageProvider
     *
     * @param storageProvider the storageProvider to use
     */
    public TaskScheduler(PartitionedStorageProvider storageProvider) {
        this(storageProvider, emptyList());
    }

    /**
     * Creates a new TaskScheduler using the provided storageProvider and the list of TaskFilters that will be used for every background task
     *
     * @param storageProvider the storageProvider to use
     * @param taskFilters      list of taskFilters that will be used for every task
     */
    public TaskScheduler(PartitionedStorageProvider storageProvider, List<TaskFilter> taskFilters) {
        this(storageProvider, new TaskDetailsAsmGenerator(), taskFilters);
    }

    public TaskScheduler(PartitionedStorageProvider storageProvider, TaskDetailsGenerator taskDetailsGenerator, List<TaskFilter> taskFilters) {
        super(storageProvider, taskFilters);
        if (taskDetailsGenerator == null)
            throw new IllegalArgumentException("A TaskDetailsGenerator is required to use the TaskScheduler.");
        this.taskDetailsGenerator = taskDetailsGenerator;
        BackgroundTask.setTaskScheduler(this);
    }

    /**
     * Creates a new fire-and-forget task based on a given lambda.
     * <h3>An example:</h3>
     * <pre>{@code
     *            MyService service = new MyService();
     *            taskScheduler.enqueue(() -> service.doWork());
     *       }</pre>
     *
     * @param task the lambda which defines the fire-and-forget task
     * @return the id of the task
     */
    public TaskId enqueue(TaskLambda task) {
        return enqueue(null, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda.
     * If a task with that id already exists, Carrot will not save it again.
     *
     * <h3>An example:</h3>
     * <pre>{@code
     *            MyService service = new MyService();
     *            taskScheduler.enqueue(id, () -> service.doWork());
     *       }</pre>
     *
     * @param id  the uuid with which to save the task
     * @param task the lambda which defines the fire-and-forget task
     * @return the id of the task
     */
    public TaskId enqueue(UUID id, TaskLambda task) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(task);
        return enqueue(id, taskDetails);
    }

    /**
     * Creates new fire-and-forget tasks for each item in the input stream using the lambda passed as {@code taskFromStream}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      Stream<UUID> workStream = getWorkStream();
     *      taskScheduler.enqueue(workStream, (uuid) -> service.doWork(uuid));
     * }</pre>
     *
     * @param input         the stream of items for which to create fire-and-forget tasks
     * @param taskFromStream the lambda which defines the fire-and-forget task to create for each item in the {@code input}
     * @param <T>           generic type for TaskLambdaFromStream
     */
    public <T> void enqueue(Stream<T> input, TaskLambdaFromStream<T> taskFromStream) {
        input
                .map(x -> taskDetailsGenerator.toTaskDetails(x, taskFromStream))
                .map(Task::new)
                .collect(batchCollector(BATCH_SIZE, this::saveTasks));
    }

    /**
     * Creates a new fire-and-forget task based on a given lambda. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *            taskScheduler.<MyService>enqueue(x -> x.doWork());
     *       }</pre>
     *
     * @param iocTask the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the task
     */
    public <S> TaskId enqueue(IocTaskLambda<S> iocTask) {
        return enqueue(null, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on a given lambda. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     *
     * <h3>An example:</h3>
     * <pre>{@code
     *            taskScheduler.<MyService>enqueue(id, x -> x.doWork());
     *       }</pre>
     *
     * @param id     the uuid with which to save the task
     * @param iocTask the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the task
     */
    public <S> TaskId enqueue(UUID id, IocTaskLambda<S> iocTask) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(iocTask);
        return enqueue(id, taskDetails);
    }

    /**
     * Creates new fire-and-forget tasks for each item in the input stream using the lambda passed as {@code taskFromStream}. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      Stream<UUID> workStream = getWorkStream();
     *      taskScheduler.<MyService, UUID>enqueue(workStream, (x, uuid) -> x.doWork(uuid));
     * }</pre>
     *
     * @param input            the stream of items for which to create fire-and-forget tasks
     * @param iocTaskFromStream the lambda which defines the fire-and-forget task to create for each item in the {@code input}
     * @param <T> generic type for IocTaskLambdaFromStream
     * @param <S> generic type for IocTaskLambdaFromStream
     */
    public <S, T> void enqueue(Stream<T> input, IocTaskLambdaFromStream<S, T> iocTaskFromStream) {
        input
                .map(x -> taskDetailsGenerator.toTaskDetails(x, iocTaskFromStream))
                .map(Task::new)
                .collect(batchCollector(BATCH_SIZE, this::saveTasks));
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(ZonedDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(ZonedDateTime zonedDateTime, TaskLambda task) {
        return schedule(null, zonedDateTime.toInstant(), task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(id, ZonedDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, ZonedDateTime zonedDateTime, TaskLambda task) {
        return schedule(id, zonedDateTime.toInstant(), task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(ZonedDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param iocTask        the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(ZonedDateTime zonedDateTime, IocTaskLambda<S> iocTask) {
        return schedule(null, zonedDateTime.toInstant(), iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(id, ZonedDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param zonedDateTime the moment in time at which the task will be enqueued.
     * @param iocTask        the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(UUID id, ZonedDateTime zonedDateTime, IocTaskLambda<S> iocTask) {
        return schedule(id, zonedDateTime.toInstant(), iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(OffsetDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param task            the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(OffsetDateTime offsetDateTime, TaskLambda task) {
        return schedule(null, offsetDateTime.toInstant(), task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(id, OffsetDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id             the uuid with which to save the task
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param task            the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, OffsetDateTime offsetDateTime, TaskLambda task) {
        return schedule(id, offsetDateTime.toInstant(), task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(OffsetDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param iocTask         the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(OffsetDateTime offsetDateTime, IocTaskLambda<S> iocTask) {
        return schedule(null, offsetDateTime.toInstant(), iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(id, OffsetDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id             the uuid with which to save the task
     * @param offsetDateTime The moment in time at which the task will be enqueued.
     * @param iocTask         the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(UUID id, OffsetDateTime offsetDateTime, IocTaskLambda<S> iocTask) {
        return schedule(id, offsetDateTime.toInstant(), iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(LocalDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(LocalDateTime localDateTime, TaskLambda task) {
        return schedule(localDateTime.atZone(systemDefault()).toInstant(), task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(id, LocalDateTime.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param task           the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, LocalDateTime localDateTime, TaskLambda task) {
        return schedule(id, localDateTime.atZone(systemDefault()).toInstant(), task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(LocalDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param iocTask        the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(LocalDateTime localDateTime, IocTaskLambda<S> iocTask) {
        return schedule(localDateTime.atZone(systemDefault()).toInstant(), iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(LocalDateTime.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id            the uuid with which to save the task
     * @param localDateTime the moment in time at which the task will be enqueued. It will use the systemDefault ZoneId to transform it to an UTC Instant
     * @param iocTask        the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(UUID id, LocalDateTime localDateTime, IocTaskLambda<S> iocTask) {
        return schedule(id, localDateTime.atZone(systemDefault()).toInstant(), iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(Instant.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param instant the moment in time at which the task will be enqueued.
     * @param task     the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(Instant instant, TaskLambda task) {
        return schedule(null, instant, task);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.schedule(id, Instant.now().plusHours(5), () -> service.doWork());
     * }</pre>
     *
     * @param id      the uuid with which to save the task
     * @param instant the moment in time at which the task will be enqueued.
     * @param task     the lambda which defines the fire-and-forget task
     * @return the id of the Task
     */
    public TaskId schedule(UUID id, Instant instant, TaskLambda task) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(task);
        return schedule(id, instant, taskDetails);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(Instant.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param instant the moment in time at which the task will be enqueued.
     * @param iocTask  the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(Instant instant, IocTaskLambda<S> iocTask) {
        return schedule(null, instant, iocTask);
    }

    /**
     * Creates a new fire-and-forget task based on the given lambda and schedules it to be enqueued at the given moment of time. The IoC container will be used to resolve {@code MyService}.
     * If a task with that id already exists, Carrot will not save it again.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>schedule(id, Instant.now().plusHours(5), x -> x.doWork());
     * }</pre>
     *
     * @param id      the uuid with which to save the task
     * @param instant the moment in time at which the task will be enqueued.
     * @param iocTask  the lambda which defines the fire-and-forget task
     * @param <S> generic type for IocTaskLambda
     * @return the id of the Task
     */
    public <S> TaskId schedule(UUID id, Instant instant, IocTaskLambda<S> iocTask) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(iocTask);
        return schedule(id, instant, taskDetails);
    }

    /**
     * Creates a new recurring task based on the given lambda and the given cron expression. The tasks will be scheduled using the systemDefault timezone.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.scheduleRecurrently(Cron.daily(), () -> service.doWork());
     * }</pre>
     *
     * @param cron The cron expression defining when to run this recurring task
     * @param task  the lambda which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public String scheduleRecurrently(String cron, TaskLambda task) {
        return scheduleRecurrently(null, cron, task);
    }

    /**
     * Creates a new recurring task based on the given cron expression and the given lambda. The IoC container will be used to resolve {@code MyService}. The tasks will be scheduled using the systemDefault timezone.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>scheduleRecurrently(Cron.daily(), x -> x.doWork());
     * }</pre>
     *
     * @param cron   The cron expression defining when to run this recurring task
     * @param iocTask the lambda which defines the recurring task
     * @param <S> generic type for IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public <S> String scheduleRecurrently(String cron, IocTaskLambda<S> iocTask) {
        return scheduleRecurrently(null, cron, iocTask);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression and lambda. The tasks will be scheduled using the systemDefault timezone
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.scheduleRecurrently("my-recurring-task", Cron.daily(), () -> service.doWork());
     * }</pre>
     *
     * @param id   the id of this recurring task which can be used to alter or delete it
     * @param cron The cron expression defining when to run this recurring task
     * @param task  the lambda which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public String scheduleRecurrently(String id, String cron, TaskLambda task) {
        return scheduleRecurrently(id, cron, systemDefault(), task);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression and lambda. The IoC container will be used to resolve {@code MyService}. The tasks will be scheduled using the systemDefault timezone
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>scheduleRecurrently("my-recurring-task", Cron.daily()),  x -> x.doWork();
     * }</pre>
     *
     * @param id     the id of this recurring task which can be used to alter or delete it
     * @param cron   The cron expression defining when to run this recurring task
     * @param iocTask the lambda which defines the recurring task
     * @param <S> generic type for IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public <S> String scheduleRecurrently(String id, String cron, IocTaskLambda<S> iocTask) {
        return scheduleRecurrently(id, cron, systemDefault(), iocTask);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression, {@code ZoneId} and lambda.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      taskScheduler.scheduleRecurrently("my-recurring-task", Cron.daily(), ZoneId.of("Europe/Brussels"), () -> service.doWork());
     * }</pre>
     *
     * @param id     the id of this recurring task which can be used to alter or delete it
     * @param cron   The cron expression defining when to run this recurring task
     * @param zoneId The zoneId (timezone) of when to run this recurring task
     * @param task    the lambda which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public String scheduleRecurrently(String id, String cron, ZoneId zoneId, TaskLambda task) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(task);
        return scheduleRecurrently(id, taskDetails, CronExpression.create(cron), zoneId);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, cron expression, {@code ZoneId} and lambda. The IoC container will be used to resolve {@code MyService}.
     * <h3>An example:</h3>
     * <pre>{@code
     *      taskScheduler.<MyService>scheduleRecurrently("my-recurring-task", Cron.daily(), ZoneId.of("Europe/Brussels"), x -> x.doWork());
     * }</pre>
     *
     * @param id     the id of this recurring task which can be used to alter or delete it
     * @param cron   The cron expression defining when to run this recurring task
     * @param zoneId The zoneId (timezone) of when to run this recurring task
     * @param iocTask the lambda which defines the recurring task
     * @param <S> generic type for IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     * @see cn.boboweike.carrot.scheduling.cron.Cron
     */
    public <S> String scheduleRecurrently(String id, String cron, ZoneId zoneId, IocTaskLambda<S> iocTask) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(iocTask);
        return scheduleRecurrently(id, taskDetails, CronExpression.create(cron), zoneId);
    }

    /**
     * Creates a new recurring task based on the given duration and the given lambda. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently(Duration.parse("P5D"), () -> service.doWork());
     * }</pre>
     *
     * @param duration the duration defining the time between each instance of this recurring task.
     * @param task      the lambda which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public String scheduleRecurrently(Duration duration, TaskLambda task) {
        return scheduleRecurrently(null, duration, task);
    }

    /**
     * Creates a new recurring task based on the given duration and the given lambda. The IoC container will be used to resolve {@code MyService}. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently(Duration.parse("P5D"), x -> x.doWork());
     * }</pre>
     *
     * @param duration the duration defining the time between each instance of this recurring task
     * @param iocTask   the lambda which defines the recurring task
     * @param <S> generic type for IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public <S> String scheduleRecurrently(Duration duration, IocTaskLambda<S> iocTask) {
        return scheduleRecurrently(null, duration, iocTask);
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, duration and lambda. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      MyService service = new MyService();
     *      BackgroundTask.scheduleRecurrently("my-recurring-task", Duration.parse("P5D"), () -> service.doWork());
     * }</pre>
     *
     * @param id       the id of this recurring task which can be used to alter or delete it
     * @param duration the duration defining the time between each instance of this recurring task
     * @param task      the lambda which defines the recurring task
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public String scheduleRecurrently(String id, Duration duration, TaskLambda task) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(task);
        return scheduleRecurrently(id, taskDetails, new Interval(duration), systemDefault());
    }

    /**
     * Creates a new or alters the existing recurring task based on the given id, duration and lambda. The IoC container will be used to resolve {@code MyService}. The first run of this recurring task will happen after the given duration unless your duration is smaller or equal than your backgroundTaskServer pollInterval.
     * <h3>An example:</h3>
     * <pre>{@code
     *      BackgroundTask.<MyService>scheduleRecurrently("my-recurring-task", Duration.parse("P5D"), x -> x.doWork());
     * }</pre>
     *
     * @param id       the id of this recurring task which can be used to alter or delete it
     * @param duration the duration defining the time between each instance of this recurring task
     * @param iocTask   the lambda which defines the recurring task
     * @param <S> generic type for IocTaskLambda
     * @return the id of this recurring task which can be used to alter or delete it
     */
    public <S> String scheduleRecurrently(String id, Duration duration, IocTaskLambda<S> iocTask) {
        TaskDetails taskDetails = taskDetailsGenerator.toTaskDetails(iocTask);
        return scheduleRecurrently(id, taskDetails, new Interval(duration), systemDefault());
    }
}

