package cn.boboweike.carrot.server;

import cn.boboweike.carrot.scheduling.exceptions.TaskNotFoundException;
import cn.boboweike.carrot.server.runner.BackgroundTaskRunner;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.context.CarrotDashboardLogger;
import cn.boboweike.carrot.tasks.filters.TaskPerformingFilters;
import cn.boboweike.carrot.tasks.mappers.MDCMapper;
import cn.boboweike.carrot.tasks.states.IllegalTaskStateChangeException;
import cn.boboweike.carrot.tasks.states.StateName;
import cn.boboweike.carrot.utils.annotations.VisibleFor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.atomic.AtomicInteger;

import static cn.boboweike.carrot.tasks.states.StateName.*;
import static cn.boboweike.carrot.utils.exceptions.Exceptions.hasCause;

public class BackgroundTaskPerformer implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackgroundTaskPerformer.class);

    private static final AtomicInteger concurrentModificationExceptionCounter = new AtomicInteger();
    private final BackgroundTaskServer backgroundTaskServer;
    private final TaskPerformingFilters taskPerformingFilters;
    private final Task task;

    public BackgroundTaskPerformer(BackgroundTaskServer backgroundTaskServer, Task task) {
        this.backgroundTaskServer = backgroundTaskServer;
        this.taskPerformingFilters = new TaskPerformingFilters(task, backgroundTaskServer.getTaskFilters());
        this.task = task;
    }

    public void run() {
        try {
            backgroundTaskServer.getTaskZooKeeper().notifyThreadOccupied();
            boolean canProcess = updateTaskStateToProcessingRunTaskFiltersAndReturnIfProcessingCanStart();
            if (canProcess) {
                runActualTask();
                updateTaskStateToSucceededAndRunTaskFilters();
            }
        } catch (Exception e) {
            if (isTaskDeletedWhileProcessing(e)) {
                // nothing to do anymore as Task is deleted
                return;
            } else if (isTaskServerStopped(e)) {
                updateTaskStateToFailedAndRunTaskFilters("Task processing was stopped as background task server has stopped", e);
                Thread.currentThread().interrupt();
            } else if (isTaskNotFoundException(e)) {
                updateTaskStateToFailedAndRunTaskFilters("Task method not found", e);
            } else {
                updateTaskStateToFailedAndRunTaskFilters("An exception occurred during the performance of the task", e);
            }
        } finally {
            backgroundTaskServer.getTaskZooKeeper().notifyThreadIdle();
        }
    }

    private boolean updateTaskStateToProcessingRunTaskFiltersAndReturnIfProcessingCanStart() {
        try {
            task.startProcessingOn(backgroundTaskServer);
            saveAndRunStateRelatedTaskFilters(task);
            LOGGER.debug("Task(id={}, taskName='{}') processing started", task.getId(), task.getTaskName());
            return task.hasState(PROCESSING);
        } catch (ConcurrentTaskModificationException e) {
            // processing already started on other server
            LOGGER.trace("Could not start processing task {} - it is already in a newer state (collision {})", task.getId(), concurrentModificationExceptionCounter.incrementAndGet());
            return false;
        }
    }

    private void runActualTask() throws Exception {
        try {
            MDCMapper.loadMDCContextFromTask(task);
            CarrotDashboardLogger.setTask(task);
            backgroundTaskServer.getTaskZooKeeper().startProcessing(task, Thread.currentThread());
            LOGGER.trace("Task(id={}, taskName='{}') is running", task.getId(), task.getTaskName());
            taskPerformingFilters.runOnTaskProcessingFilters();
            BackgroundTaskRunner backgroundTaskRunner = backgroundTaskServer.getBackgroundTaskRunner(task);
            backgroundTaskRunner.run(task);
            taskPerformingFilters.runOnTaskProcessedFilters();
        } finally {
            backgroundTaskServer.getTaskZooKeeper().stopProcessing(task);
            CarrotDashboardLogger.clearTask();
            MDC.clear();
        }
    }

    private void updateTaskStateToSucceededAndRunTaskFilters() {
        try {
            LOGGER.debug("Task(id={}, taskName='{}') processing succeeded", task.getId(), task.getTaskName());
            task.succeeded();
            saveAndRunStateRelatedTaskFilters(task);
        } catch (IllegalTaskStateChangeException ex) {
            if (ex.getFrom() == DELETED) {
                LOGGER.info("Task finished successfully but it was already deleted - ignoring illegal state change from {} to {}", ex.getFrom(), ex.getTo());
            } else {
                throw ex;
            }
        } catch (Exception badException) {
            LOGGER.error("ERROR - could not update task(id={}, taskName='{}') to SUCCEEDED state", task.getId(), task.getTaskName(), badException);
        }
    }

    private void updateTaskStateToFailedAndRunTaskFilters(String message, Exception e) {
        try {
            Exception actualException = unwrapException(e);
            task.failed(message, actualException);
            saveAndRunStateRelatedTaskFilters(task);
            if (task.getState() == FAILED) {
                LOGGER.error("Task(id={}, taskName='{}') processing failed: {}", task.getId(), task.getTaskName(), message, actualException);
            } else {
                LOGGER.warn("Task(id={}, taskName='{}') processing failed: {}", task.getId(), task.getTaskName(), message, actualException);
            }
        } catch (IllegalTaskStateChangeException ex) {
            if (ex.getFrom() == DELETED) {
                LOGGER.info("Task processing failed but it was already deleted - ignoring illegal state change from {} to {}", ex.getFrom(), ex.getTo());
            } else {
                throw ex;
            }
        } catch (Exception badException) {
            LOGGER.error("ERROR - could not update task(id={}, taskName='{}') to FAILED state", task.getId(), task.getTaskName(), badException);
        }
    }

    protected void saveAndRunStateRelatedTaskFilters(Task task) {
        taskPerformingFilters.runOnStateAppliedFilters();
        StateName beforeStateElection = task.getState();
        taskPerformingFilters.runOnStateElectionFilter();
        StateName afterStateElection = task.getState();
        this.backgroundTaskServer.getStorageProvider().saveByPartition(task, getPartition());
        if (beforeStateElection != afterStateElection) {
            taskPerformingFilters.runOnStateAppliedFilters();
        }
    }

    private Integer getPartition() {
        return this.backgroundTaskServer.getPartition();
    }

    private boolean isTaskDeletedWhileProcessing(Exception e) {
        return hasCause(e, InterruptedException.class) && task.hasState(StateName.DELETED);
    }

    private boolean isTaskServerStopped(Exception e) {
        return hasCause(e, InterruptedException.class) && !task.hasState(StateName.DELETED);
    }

    private boolean isTaskNotFoundException(Exception e) {
        return e instanceof TaskNotFoundException;
    }

    /**
     * Carrot uses reflection to run tasks. Any error in tasks is wrapped in {@link InvocationTargetException}.
     * Task details shows {@link InvocationTargetException} and its stacktrace on UI
     * with lots of internal details not related to the task.
     * It makes harder for users to read exceptions
     * and leaves less space for the actual errors' stacktraces on UI.
     */
    @VisibleFor("testing")
    static Exception unwrapException(Exception e) {
        if (e instanceof InvocationTargetException && e.getCause() instanceof Exception) {
            return (Exception) e.getCause();
        }

        return e;
    }
}
