package cn.boboweike.carrot.server;

import cn.boboweike.carrot.server.dashboard.DashboardNotificationManager;
import cn.boboweike.carrot.server.jmx.BackgroundTaskServerMBean;
import cn.boboweike.carrot.server.jmx.TaskServerStats;
import cn.boboweike.carrot.server.runner.*;
import cn.boboweike.carrot.server.strategy.WorkDistributionStrategy;
import cn.boboweike.carrot.server.tasks.CheckIfAllTasksExistTask;
import cn.boboweike.carrot.server.tasks.CreateClusterIdIfNotExists;
import cn.boboweike.carrot.server.threadpool.CarrotExecutor;
import cn.boboweike.carrot.server.threadpool.ScheduledThreadPoolCarrotExecutor;
import cn.boboweike.carrot.storage.BackgroundTaskServerStatus;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.storage.ThreadSafePartitionedStorageProvider;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.filters.TaskDefaultFilters;
import cn.boboweike.carrot.tasks.filters.TaskFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.List;
import java.util.ServiceLoader;
import java.util.Spliterator;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import static cn.boboweike.carrot.CarrotException.problematicConfigurationException;
import static cn.boboweike.carrot.server.BackgroundTaskServerConfiguration.usingStandardBackgroundTaskServerConfiguration;
import static cn.boboweike.carrot.utils.TaskUtils.assertTaskExists;
import static java.lang.Integer.compare;
import static java.util.Arrays.asList;
import static java.util.Spliterators.spliteratorUnknownSize;
import static java.util.stream.StreamSupport.stream;

public class BackgroundTaskServer implements BackgroundTaskServerMBean {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackgroundTaskServer.class);

    public static final Integer NO_PARTITION = -1;

    private final UUID backgroundTaskServerId;
    private final BackgroundTaskServerConfiguration configuration;
    private final PartitionedStorageProvider storageProvider;
    private final DashboardNotificationManager dashboardNotificationManager;
    private final List<BackgroundTaskRunner> backgroundTaskRunners;
    private final TaskDefaultFilters taskDefaultFilters;
    private final TaskServerStats taskServerStats;
    private final WorkDistributionStrategy workDistributionStrategy;
    private final ServerZooKeeper serverZooKeeper;
    private final TaskZooKeeper taskZooKeeper;
    private final BackgroundTaskServerLifecycleLock lifecycleLock;
    private volatile Instant firstHeartbeat;
    private volatile boolean isRunning;
    private volatile boolean isStopping;
    private volatile Integer partition;
    private volatile ScheduledThreadPoolExecutor zookeeperThreadPool;
    private CarrotExecutor taskExecutor;

    public BackgroundTaskServer(PartitionedStorageProvider storageProvider) {
        this(storageProvider, null);
    }

    public BackgroundTaskServer(PartitionedStorageProvider storageProvider, TaskActivator taskActivator) {
        this(storageProvider, taskActivator, usingStandardBackgroundTaskServerConfiguration());
    }

    public BackgroundTaskServer(PartitionedStorageProvider storageProvider, TaskActivator taskActivator, BackgroundTaskServerConfiguration configuration) {
        if (storageProvider == null)
            throw new IllegalArgumentException("A StorageProvider is required to use a BackgroundTaskServer. Please see the documentation on how to setup a task StorageProvider.");

        this.backgroundTaskServerId = UUID.randomUUID();
        this.configuration = configuration;
        this.storageProvider = new ThreadSafePartitionedStorageProvider(storageProvider);
        this.dashboardNotificationManager = new DashboardNotificationManager(backgroundTaskServerId, storageProvider);
        this.backgroundTaskRunners = initializeBackgroundTaskRunners(taskActivator);
        this.taskDefaultFilters = new TaskDefaultFilters();
        this.taskServerStats = new TaskServerStats();
        this.workDistributionStrategy = createWorkDistributionStrategy(configuration);
        this.serverZooKeeper = createServerZooKeeper();
        this.taskZooKeeper = createTaskZooKeeper();
        this.lifecycleLock = new BackgroundTaskServerLifecycleLock();
    }

    public UUID getId() {
        return backgroundTaskServerId;
    }

    public void start() {
        start(true);
    }

    public void start(boolean guard) {
        if (guard) {
            if (isStarted()) return;
            try (BackgroundTaskServerLifecycleLock ignored = lifecycleLock.lock()) {
                if (isStarted()) return; // double check
                firstHeartbeat = Instant.now();
                isRunning = true;
                isStopping = false;
                startZooKeepers();
                startWorkers();
                runStartupTasks();
            }
        }
    }

    public void pauseProcessing() {
        if (isStopped()) throw new IllegalStateException("First start the BackgroundTaskServer before pausing");
        if (isPaused()) return;
        try (BackgroundTaskServerLifecycleLock ignored = lifecycleLock.lock()) {
            isRunning = false;
            stopWorkers();
            LOGGER.info("Paused task processing");
        }
    }

    public void resumeProcessing() {
        if (isStopped()) throw new IllegalStateException("First start the BackgroundTaskServer before resuming");
        if (isProcessing()) return;
        try (BackgroundTaskServerLifecycleLock ignored = lifecycleLock.lock()) {
            startWorkers();
            isRunning = true;
            LOGGER.info("Resumed task processing");
        }
    }

    public void stop() {
        if (isStopped()) return;
        try (BackgroundTaskServerLifecycleLock ignored = lifecycleLock.lock()) {
            if (isStopped()) return; // double check
            isStopping = true;
            LOGGER.info("BackgroundTaskServer and BackgroundTaskPerformers - stopping (waiting for all tasks to complete - max 10 seconds)");
            stopWorkers();
            stopZooKeepers();
            // need to stopZooKeepers first, then set partition = null
            // since stopZooKeepers will call serverZooKeeper.stop() and it will unlock the partition first
            partition = null;
            isRunning = false;
            firstHeartbeat = null;
            LOGGER.info("BackgroundTaskServer and BackgroundTaskPerformers stopped");
            isStopping = false;
        }
    }

    public boolean isAnnounced() { // no lock here to avoid possible deadlock when stop is called
        return partition != null;
    }

    public boolean isUnAnnounced() {
        return !isAnnounced();
    }

    public Integer getPartition() {
        return this.partition;
    }

    void setPartition(Integer partition) {
        if (isStopped()) return;

        this.partition = partition;
        if (partition != null && partition != NO_PARTITION) {
            LOGGER.info("Carrot BackgroundTaskServer ({}) using {} and {} BackgroundTaskPerformers started successfully, " +
                    "acquired partition is {}.", getId(), storageProvider.getName(), workDistributionStrategy.getWorkerCount(), this.partition);
        } else {
            LOGGER.warn("Carrot BackgroundTaskServer failed to acquire partition");
        }
    }

    public boolean isRunning() {
        try (BackgroundTaskServerLifecycleLock ignored = lifecycleLock.lock()) {
            return isRunning;
        }
    }

    public BackgroundTaskServerStatus getServerStatus() {
        return new BackgroundTaskServerStatus(
                backgroundTaskServerId, workDistributionStrategy.getWorkerCount(),
                configuration.pollIntervalInSeconds, configuration.deleteSucceededTasksAfter, configuration.permanentlyDeleteDeletedTasksAfter,
                firstHeartbeat, Instant.now(), isRunning, taskServerStats.getSystemTotalMemory(), taskServerStats.getSystemFreeMemory(),
                taskServerStats.getSystemCpuLoad(), taskServerStats.getProcessMaxMemory(), taskServerStats.getProcessFreeMemory(),
                taskServerStats.getProcessAllocatedMemory(), taskServerStats.getProcessCpuLoad(), this.partition
        );
    }

    public TaskZooKeeper getTaskZooKeeper() {
        return taskZooKeeper;
    }

    public PartitionedStorageProvider getStorageProvider() {
        return storageProvider;
    }

    public BackgroundTaskServerConfiguration getConfiguration() {
        return configuration;
    }

    public DashboardNotificationManager getDashboardNotificationManager() {
        return dashboardNotificationManager;
    }

    public WorkDistributionStrategy getWorkDistributionStrategy() {
        return workDistributionStrategy;
    }

    public void setTaskFilters(List<TaskFilter> taskFilters) {
        this.taskDefaultFilters.addAll(taskFilters);
    }

    public TaskDefaultFilters getTaskFilters() {
        return taskDefaultFilters;
    }

    BackgroundTaskRunner getBackgroundTaskRunner(Task task) {
        assertTaskExists(task.getTaskDetails());
        return backgroundTaskRunners.stream()
                .filter(taskRunner -> taskRunner.supports(task))
                .findFirst()
                .orElseThrow(() -> problematicConfigurationException("Could not find a BackgroundTaskRunner: either no TaskActivator is registered, your Background Task Class is not registered within the IoC container or your Task does not have a default no-arg constructor."));
    }

    void processTask(Task task) {
        BackgroundTaskPerformer backgroundTaskPerformer = new BackgroundTaskPerformer(this, task);
        taskExecutor.execute(backgroundTaskPerformer);
        LOGGER.debug("Submitted BackgroundTaskPerformer for task {} to executor service", task.getId());
    }

    boolean isStarted() {
        return !isStopped();
    }

    boolean isStopped() { // no lock here to avoid possible deadlock when stop is called
        return zookeeperThreadPool == null;
    }

    public boolean isStoppingOrStopped() { // no lock here to avoid possible deadlock when stop is called
        return isStopping || zookeeperThreadPool == null;
    }

    boolean isPaused() {
        return !isProcessing();
    }

    boolean isProcessing() {
        try (BackgroundTaskServerLifecycleLock ignored = lifecycleLock.lock()) {
            return isRunning;
        }
    }

    private void startZooKeepers() {
        zookeeperThreadPool = new ScheduledThreadPoolCarrotExecutor(2, "backgroundtask-zookeeper-pool");
        // why fixedDelay: in case of long stop-the-world garbage collections, the zookeeper tasks will queue up
        // and all will be launched one after another
        zookeeperThreadPool.scheduleWithFixedDelay(serverZooKeeper, 0, configuration.pollIntervalInSeconds, TimeUnit.SECONDS);
        zookeeperThreadPool.scheduleWithFixedDelay(taskZooKeeper, 1, configuration.pollIntervalInSeconds, TimeUnit.SECONDS);
    }

    private void stopZooKeepers() {
        serverZooKeeper.stop();
        stop(this.zookeeperThreadPool);
        this.zookeeperThreadPool = null;
    }

    private void startWorkers() {
        taskExecutor = loadCarrotExecutor();
        taskExecutor.start();
    }

    private void stopWorkers() {
        if (taskExecutor == null) return;
        taskExecutor.stop();
        this.taskExecutor = null;
    }

    private void runStartupTasks() {
        try {
            List<Runnable> startupTasks = asList(
                    new CreateClusterIdIfNotExists(this),
                    new CheckIfAllTasksExistTask(this));
            startupTasks.forEach(taskExecutor::execute);
        } catch (Exception notImportant) {
            // server is shut down immediately
        }
    }

    private List<BackgroundTaskRunner> initializeBackgroundTaskRunners(TaskActivator taskActivator) {
        return asList(
                new BackgroundTaskWithIocRunner(taskActivator),
                new BackgroundTaskWithoutIocRunner(),
                new BackgroundStaticTaskWithoutIocRunner(),
                new BackgroundStaticFieldTaskWithoutIocRunner()
        );
    }

    private void stop(ScheduledExecutorService executorService) {
        if (executorService == null) return;
        executorService.shutdown();
        try {
            if (!executorService.awaitTermination(10, TimeUnit.SECONDS)) {
                LOGGER.info("Carrot BackgroundTaskServer shutdown requested - waiting for tasks to finish (at most 10 seconds)");
                executorService.shutdownNow();
                if (!executorService.awaitTermination(10, TimeUnit.SECONDS)) {
                    System.err.println("Pool did not terminate");
                }
            }
        } catch (InterruptedException e) {
            executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private ServerZooKeeper createServerZooKeeper() {
        return new ServerZooKeeper(this);
    }

    private TaskZooKeeper createTaskZooKeeper() {
        return new TaskZooKeeper(this);
    }

    private WorkDistributionStrategy createWorkDistributionStrategy(BackgroundTaskServerConfiguration configuration) {
        return configuration.backgroundTaskServerWorkerPolicy.toWorkDistributionStrategy(this);
    }

    private CarrotExecutor loadCarrotExecutor() {
        ServiceLoader<CarrotExecutor> serviceLoader = ServiceLoader.load(CarrotExecutor.class);
        return stream(spliteratorUnknownSize(serviceLoader.iterator(), Spliterator.ORDERED), false)
                .sorted((a, b) -> compare(b.getPriority(), a.getPriority()))
                .findFirst()
                .orElse(new ScheduledThreadPoolCarrotExecutor(workDistributionStrategy.getWorkerCount(), "backgroundtask-worker-pool"));
    }

    private static class BackgroundTaskServerLifecycleLock implements AutoCloseable {
        private final ReentrantLock reentrantLock = new ReentrantLock();

        public BackgroundTaskServerLifecycleLock lock() {
            if (reentrantLock.isHeldByCurrentThread()) {
                return this;
            }

            reentrantLock.lock();
            return this;
        }

        @Override
        public void close() {
            reentrantLock.unlock();
        }
    }
}
