package cn.boboweike.carrot.server;

import cn.boboweike.carrot.server.configuration.*;

import java.time.Duration;

/**
 * This class allows to configure the BackgroundTaskServer
 */
public class BackgroundTaskServerConfiguration {
    public static final int DEFAULT_POLL_INTERVAL_IN_SECONDS = 15;
    public static final Duration DEFAULT_DELETE_SUCCEEDED_TASKS_DURATION = Duration.ofHours(36);
    public static final Duration DEFAULT_PERMANENTLY_DELETE_TASKS_DURATION = Duration.ofHours(72);

    int pollIntervalInSeconds = DEFAULT_POLL_INTERVAL_IN_SECONDS;
    Duration deleteSucceededTasksAfter = DEFAULT_DELETE_SUCCEEDED_TASKS_DURATION;
    Duration permanentlyDeleteDeletedTasksAfter = DEFAULT_PERMANENTLY_DELETE_TASKS_DURATION;
    BackgroundTaskServerWorkerPolicy backgroundTaskServerWorkerPolicy = new DefaultBackgroundTaskServerWorkerPolicy();
    ConcurrentTaskModificationPolicy concurrentTaskModificationPolicy = new DefaultConcurrentTaskModificationPolicy();

    private BackgroundTaskServerConfiguration() {

    }

    /**
     * This returns the default configuration with the BackgroundTaskServer with a poll interval of 15 seconds and a worker count based on the CPU
     *
     * @return the default CarrotDashboard configuration
     */
    public static BackgroundTaskServerConfiguration usingStandardBackgroundTaskServerConfiguration() {
        return new BackgroundTaskServerConfiguration();
    }

    /**
     * Allows to set the pollIntervalInSeconds for the BackgroundTaskServer
     *
     * @param pollIntervalInSeconds the pollIntervalInSeconds
     * @return the same configuration instance which provides a fluent api
     */
    public BackgroundTaskServerConfiguration andPollIntervalInSeconds(int pollIntervalInSeconds) {
        if (pollIntervalInSeconds < 5)
            throw new IllegalArgumentException("The pollIntervalInSeconds can not be smaller than 5 - otherwise it will cause too much load on your SQL/noSQL datastore.");
        this.pollIntervalInSeconds = pollIntervalInSeconds;
        return this;
    }

    /**
     * Allows to set the workerCount for the BackgroundTaskServer which defines the maximum number of tasks that will be run in parallel
     *
     * @param workerCount the workerCount for the BackgroundTaskServer
     * @return the same configuration instance which provides a fluent api
     */
    public BackgroundTaskServerConfiguration andWorkerCount(int workerCount) {
        this.backgroundTaskServerWorkerPolicy = new FixedSizeBackgroundTaskServerWorkerPolicy(workerCount);
        return this;
    }

    /**
     * Allows to set the backgroundTaskServerWorkerPolicy for the BackgroundTaskServer. The backgroundTaskServerWorkerPolicy will determine
     * the final WorkDistributionStrategy used by the BackgroundTaskServer.
     *
     * @param backgroundTaskServerWorkerPolicy the backgroundTaskServerWorkerPolicy
     * @return the same configuration instance which provides a fluent api
     */
    public BackgroundTaskServerConfiguration andBackgroundTaskServerWorkerPolicy(BackgroundTaskServerWorkerPolicy backgroundTaskServerWorkerPolicy) {
        this.backgroundTaskServerWorkerPolicy = backgroundTaskServerWorkerPolicy;
        return this;
    }

    /**
     * Allows to set the duration to wait before deleting succeeded tasks
     *
     * @param duration the duration to wait before deleting successful tasks
     * @return the same configuration instance which provides a fluent api
     */
    public BackgroundTaskServerConfiguration andDeleteSucceededTasksAfter(Duration duration) {
        this.deleteSucceededTasksAfter = duration;
        return this;
    }

    /**
     * Allows to set the duration to wait before permanently deleting succeeded tasks
     *
     * @param duration the duration to wait before permanently deleting successful tasks
     * @return the same configuration instance which provides a fluent api
     */
    public BackgroundTaskServerConfiguration andPermanentlyDeleteDeletedTasksAfter(Duration duration) {
        this.permanentlyDeleteDeletedTasksAfter = duration;
        return this;
    }

    /**
     * Allows to set the ConcurrentTaskModificationPolicy for the BackgroundTaskServer. The ConcurrentTaskModificationPolicy will determine
     * how the BackgroundTaskServer will react to concurrent modifications the tasks.
     * <p>
     * Use with care.
     *
     * @param concurrentTaskModificationPolicy the concurrentTaskModificationPolicy
     * @return the same configuration instance which provides a fluent api
     */
    public BackgroundTaskServerConfiguration andConcurrentTaskModificationPolicy(ConcurrentTaskModificationPolicy concurrentTaskModificationPolicy) {
        this.concurrentTaskModificationPolicy = concurrentTaskModificationPolicy;
        return this;
    }
}
