package cn.boboweike.carrot.server;

import cn.boboweike.carrot.server.dashboard.CpuAllocationIrregularityNotification;
import cn.boboweike.carrot.server.dashboard.DashboardNotificationManager;
import cn.boboweike.carrot.storage.BackgroundTaskServerStatus;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.storage.ServerTimedOutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static cn.boboweike.carrot.server.BackgroundTaskServer.NO_PARTITION;

public class ServerZooKeeper implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServerZooKeeper.class);

    private final BackgroundTaskServer backgroundTaskServer;
    private final PartitionedStorageProvider storageProvider;
    private final DashboardNotificationManager dashboardNotificationManager;
    private final int timeoutInSeconds;
    private final int lockDurationInSeconds;
    private final String lockedBy;
    private Instant lastSignalAlive;
    private Instant lastServerTimeoutCheck;
    private AtomicInteger restartAttempts;
    private final AtomicInteger extendLockFailureCount;

    public ServerZooKeeper(BackgroundTaskServer backgroundTaskServer) {
        this.backgroundTaskServer = backgroundTaskServer;
        this.storageProvider = backgroundTaskServer.getStorageProvider();
        this.dashboardNotificationManager = backgroundTaskServer.getDashboardNotificationManager();
        this.timeoutInSeconds = backgroundTaskServer.getServerStatus().getPollIntervalInSeconds() * 4;
        this.lockDurationInSeconds = backgroundTaskServer.getServerStatus().getPollIntervalInSeconds() * 4;
        this.lockedBy = backgroundTaskServer.getServerStatus().getId().toString();
        this.lastSignalAlive = Instant.now();
        this.lastServerTimeoutCheck = Instant.now();
        this.restartAttempts = new AtomicInteger();
        this.extendLockFailureCount = new AtomicInteger();
    }

    @Override
    public void run() {
        if (backgroundTaskServer.isStoppingOrStopped()) return;
        try {
            if (backgroundTaskServer.isUnAnnounced()) {
                announceBackgroundTaskServer();
            } else {
                signalBackgroundTaskServerAliveAndDoZooKeeping();
            }
            acquireOrExtendPartition();
        } catch (Exception shouldNotHappen) {
            if (restartAttempts.incrementAndGet() < 3) {
                LOGGER.error("An unrecoverable error occurred, try next run. Restart attempt: " + restartAttempts + " out of 3", shouldNotHappen);
            } else {
                LOGGER.error("An unrecoverable error occurred, restart server.", shouldNotHappen);
                restartAttempts.set(0);
                backgroundTaskServer.setPartition(null);
                new Thread(this::resetServer).start();
            }
        }
    }

    private void acquireOrExtendPartition() {
        Integer partition = backgroundTaskServer.getPartition();
        if (partition == null || partition == NO_PARTITION) {
            acquirePartition();
        } else {
            extendPartition(partition);
        }
    }

    public synchronized void stop() {
        try {
            storageProvider.signalBackgroundTaskServerStopped(backgroundTaskServer.getServerStatus());
            Integer partition = this.backgroundTaskServer.getPartition();
            if (partition != null && partition != NO_PARTITION) {
                storageProvider.unlockByPartition(partition);
                LOGGER.info("Carrot server {} unlocked partition {}", lockedBy, partition);
            }
        } catch (Exception e) {
            LOGGER.error("Error when signalling that BackgroundTaskServer stopped", e);
        }
    }

    private void announceBackgroundTaskServer() {
        final BackgroundTaskServerStatus serverStatus = backgroundTaskServer.getServerStatus();
        storageProvider.announceBackgroundTaskServer(serverStatus);
        lastSignalAlive = serverStatus.getLastHeartbeat();
    }

    private void signalBackgroundTaskServerAliveAndDoZooKeeping() {
        try {
            signalBackgroundTaskServerAlive();
            deleteServersThatTimedOut();
        } catch (ServerTimedOutException e) {
            LOGGER.error("SEVERE ERROR - Server timed out while it's still alive. Are all servers using NTP and in the same timezone? Are you having long GC cycles? Resetting server...", e);
            new Thread(this::resetServer).start();
        }
    }

    private void signalBackgroundTaskServerAlive() {
        // TODO: stop server if requested?
        final BackgroundTaskServerStatus serverStatus = backgroundTaskServer.getServerStatus();
        storageProvider.signalBackgroundTaskServerAlive(serverStatus);
        cpuAllocationIrregularity(lastSignalAlive, serverStatus.getLastHeartbeat()).ifPresent(amountOfSeconds ->
                dashboardNotificationManager.notify(new CpuAllocationIrregularityNotification(amountOfSeconds)));
        lastSignalAlive = serverStatus.getLastHeartbeat();
    }

    private void  deleteServersThatTimedOut() {
        if (Instant.now().isAfter(this.lastServerTimeoutCheck.plusSeconds(timeoutInSeconds))) {
            final Instant now = Instant.now();
            final Instant defaultTimeoutInstant = now.minusSeconds(timeoutInSeconds);
            final Instant timedOutInstantUsingLastSignalAlive = lastSignalAlive.minusMillis(500);
            final Instant timedOutInstant = min(defaultTimeoutInstant, timedOutInstantUsingLastSignalAlive);

            final int amountOfServersThatTimedOut = storageProvider.removeTimedOutBackgroundTaskServers(timedOutInstant);
            if (amountOfServersThatTimedOut > 0) {
                LOGGER.info("Removed {} server(s) that timed out", amountOfServersThatTimedOut);
            }
            this.lastServerTimeoutCheck = now;
        }
    }

    private void extendPartition(Integer partition) {
        boolean extendSuccess =
                storageProvider.extendLockByPartition(partition, lockDurationInSeconds, lockedBy);
        if (extendSuccess) {
            extendLockFailureCount.set(0);
            LOGGER.info("Carrot server {} extended partition {} for {} seconds", lockedBy, partition, lockDurationInSeconds);
        } else {
            if (extendLockFailureCount.incrementAndGet() == 3) {
                this.backgroundTaskServer.setPartition(NO_PARTITION);
                LOGGER.info("Carrot server {} failed to extend partition {} for {} seconds", lockedBy, partition, lockDurationInSeconds);
                extendLockFailureCount.set(0);
            }
        }
    }

    private void acquirePartition() {
        List<Integer> partitionList = getShuffledPartitionList(storageProvider.getTotalNumOfPartitions());
        for(Integer partition : partitionList) {
            boolean lockSuccess = storageProvider.lockByPartition(partition, lockDurationInSeconds, lockedBy);
            if (lockSuccess) {
                LOGGER.info("Carrot server {} acquired partition {} for {} seconds", lockedBy, partition, lockDurationInSeconds);
                this.backgroundTaskServer.setPartition(partition);
                return;
            }
            sleepRandomly(100, 200);
        }
        LOGGER.info("Carrot server {} failed to acquire partition for {} seconds", lockedBy, lockDurationInSeconds);
        this.backgroundTaskServer.setPartition(NO_PARTITION);
    }

    private void sleepRandomly(int low, int high) {
        try {
            long timeout = new Random().nextInt(high - low) + low;
            TimeUnit.MILLISECONDS.sleep(timeout);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    private List<Integer> getShuffledPartitionList(int totalNumberOfPartitions) {
        List<Integer> partitionList = IntStream.range(0, totalNumberOfPartitions).boxed().collect(Collectors.toList());
        Collections.shuffle(partitionList);
        return partitionList;
    }

    private void resetServer() {
        backgroundTaskServer.stop();
        backgroundTaskServer.start();
    }

    private static Instant min(Instant instant1, Instant instant2) {
        Instant[] instants = new Instant[]{instant1, instant2};
        Arrays.sort(instants);
        return instants[0];
    }

    private Optional<Integer> cpuAllocationIrregularity(Instant lastSignalAlive, Instant lastHeartbeat) {
        final Instant now = Instant.now();
        final int amount1OfSec = (int) Math.abs(lastHeartbeat.getEpochSecond() - lastSignalAlive.getEpochSecond());
        final int amount2OfSec = (int) (now.getEpochSecond() - lastSignalAlive.getEpochSecond());
        final int amount3OfSec = (int) (now.getEpochSecond() - lastHeartbeat.getEpochSecond());

        final int max = Math.max(amount1OfSec, Math.max(amount2OfSec, amount3OfSec));
        if (max > backgroundTaskServer.getServerStatus().getPollIntervalInSeconds() * 2L) {
            return Optional.of(max);
        }
        return Optional.empty();
    }
}
