package cn.boboweike.carrot.server.concurrent;

import cn.boboweike.carrot.storage.TaskNotFoundException;
import cn.boboweike.carrot.server.TaskZooKeeper;
import cn.boboweike.carrot.server.concurrent.statechanges.*;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.Task;

import java.util.Arrays;
import java.util.List;

import static java.util.stream.Collectors.toList;

/**
 * Default implementation of {@link ConcurrentTaskModificationResolver}.
 * <p>
 * If Tasks are deleted, the {@link DefaultConcurrentTaskModificationResolver} will resolve the concurrent task modification
 * by stopping the processing of the task. For other concurrent modifications, the {@link DefaultConcurrentTaskModificationResolver} will
 * throw {@link UnresolvableConcurrentTaskModificationException} as these may point to programming errors (Carrot was conceived with the idea that once a
 * task is being processed, it should not be modified anymore).
 */
public class DefaultConcurrentTaskModificationResolver implements ConcurrentTaskModificationResolver {
    private final PartitionedStorageProvider storageProvider;
    private final List<AllowedConcurrentStateChange> allowedConcurrentStateChanges;

    public DefaultConcurrentTaskModificationResolver(PartitionedStorageProvider storageProvider, TaskZooKeeper taskZooKeeper) {
        this.storageProvider = storageProvider;
        allowedConcurrentStateChanges = Arrays.asList(
                new PermanentlyDeletedWhileProcessingConcurrentStateChange(taskZooKeeper),
                new DeletedWhileProcessingConcurrentStateChange(taskZooKeeper),
                new DeletedWhileSucceededConcurrentStateChange(),
                new DeletedWhileFailedConcurrentStateChange(),
                new DeletedWhileEnqueuedConcurrentStateChange(),
                new DeletedWhileScheduledConcurrentStateChange()
        );
    }

    public void resolve(ConcurrentTaskModificationException e) {
        final List<Task> concurrentUpdatedTasks = e.getConcurrentUpdatedTasks();
        final List<ConcurrentTaskModificationResolveResult> failedToResolve = concurrentUpdatedTasks
                .stream()
                .map(this::resolve)
                .filter(ConcurrentTaskModificationResolveResult::failed)
                .collect(toList());

        if (!failedToResolve.isEmpty()) {
            throw new UnresolvableConcurrentTaskModificationException(failedToResolve);
        }
    }

    public ConcurrentTaskModificationResolveResult resolve(final Task localTask) {
        final Task taskFromStorage = getTaskFromStorageProvider(localTask);
        return allowedConcurrentStateChanges
                .stream()
                .filter(allowedConcurrentStateChange -> allowedConcurrentStateChange.matches(localTask, taskFromStorage))
                .findFirst()
                .map(allowedConcurrentStateChange -> allowedConcurrentStateChange.resolve(localTask, taskFromStorage))
                .orElse(ConcurrentTaskModificationResolveResult.failed(localTask, taskFromStorage));
    }


    private Task getTaskFromStorageProvider(Task localTask) {
        try {
            return storageProvider.getTaskById(localTask.getId());
        } catch (TaskNotFoundException e) {
            // this happens when the task was permanently deleted while processing
            return null;
        }
    }

}
