package cn.boboweike.carrot.server.concurrent;

import cn.boboweike.carrot.SevereCarrotException;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.states.ProcessingState;
import cn.boboweike.carrot.tasks.states.TaskState;
import cn.boboweike.carrot.utils.diagnostics.DiagnosticsBuilder;

import java.util.List;

import static cn.boboweike.carrot.utils.diagnostics.DiagnosticsBuilder.diagnostics;
import static java.util.stream.Collectors.toList;

public class UnresolvableConcurrentTaskModificationException extends ConcurrentTaskModificationException implements SevereCarrotException.DiagnosticsAware {
    private final List<ConcurrentTaskModificationResolveResult> concurrentTaskModificationResolveResults;

    public UnresolvableConcurrentTaskModificationException(List<ConcurrentTaskModificationResolveResult> concurrentTaskModificationResolveResults) {
        super(concurrentTaskModificationResolveResults.stream().map(ConcurrentTaskModificationResolveResult::getLocalTask).collect(toList()));
        this.concurrentTaskModificationResolveResults = concurrentTaskModificationResolveResults;
    }

    @Override
    public DiagnosticsBuilder getDiagnosticsInfo() {
        return diagnostics()
                .withTitle("Concurrent modified tasks:")
                .with(concurrentTaskModificationResolveResults, ((resolveResult, diagnosticsBuilder) -> appendDiagnosticsInfo(diagnosticsBuilder, resolveResult)));
    }

    private void appendDiagnosticsInfo(DiagnosticsBuilder diagnostics, ConcurrentTaskModificationResolveResult resolveResult) {
        Task localTask = resolveResult.getLocalTask();
        Task taskFromStorage = resolveResult.getTaskFromStorage();

        diagnostics
                .withLine("Task id: " + localTask.getId())
                .withIndentedLine("Local version: " + localTask.getVersion() + "; Storage version: " + taskFromStorage.getVersion())
                .withIndentedLine("Local state: " + getTaskStates(localTask))
                .withIndentedLine("Storage state: " + getTaskStates(taskFromStorage));
    }

    private String getTaskStates(Task task) {
        StringBuilder result = new StringBuilder();
        final int taskStatesToShow = Math.min(3, task.getTaskStates().size());
        for (int i = 1; i <= taskStatesToShow; i++) {
            final TaskState taskState = task.getTaskState(-i);
            result.append(taskState.getName());
            result.append(" (at " + taskState.getUpdatedAt());
            if (taskState instanceof ProcessingState) {
                result.append(" on BackgroundTaskServer " + ((ProcessingState) taskState).getServerId());
            }
            result.append(")");
            if (i < taskStatesToShow) {
                result.append(" ← ");
            }
        }
        return result.toString();
    }
}
