package cn.boboweike.carrot.server.concurrent;

import cn.boboweike.carrot.server.TaskZooKeeper;
import cn.boboweike.carrot.storage.ConcurrentTaskModificationException;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.states.IllegalTaskStateChangeException;
import cn.boboweike.carrot.utils.annotations.Beta;

import java.util.List;

/**
 * A new implementation of {@link ConcurrentTaskModificationResolver} that will always use the latest version of the task found in the database (SQL or NoSQL).
 * <p>
 * Only use this {@link ConcurrentTaskModificationResolver} if you are altering tasks after they started processing (e.g. e.g. deleting, rescheduling, ...).
 * Using this {@link ConcurrentTaskModificationResolver} can result in the same task being executed multiple times.
 */
@Beta
public class UseStorageProviderTaskConcurrentTaskModificationResolver implements ConcurrentTaskModificationResolver {
    private final TaskZooKeeper taskZooKeeper;

    public UseStorageProviderTaskConcurrentTaskModificationResolver(TaskZooKeeper taskZooKeeper) {
        this.taskZooKeeper = taskZooKeeper;
    }

    @Override
    public void resolve(ConcurrentTaskModificationException e) {
        final List<Task> concurrentUpdatedTasks = e.getConcurrentUpdatedTasks();
        concurrentUpdatedTasks
                .forEach(task -> resolve(task, e));
    }

    public ConcurrentTaskModificationResolveResult resolve(final Task localTask, ConcurrentTaskModificationException e) {
        failLocalIfPossible(localTask, e);
        final Thread threadProcessingTask = taskZooKeeper.getThreadProcessingTask(localTask);
        if (threadProcessingTask != null) {
            threadProcessingTask.interrupt();
        }
        return ConcurrentTaskModificationResolveResult.succeeded(localTask);
    }

    private void failLocalIfPossible(Task localTask, ConcurrentTaskModificationException e) {
        try {
            localTask.failed("Task is already updated in StorageProvider, discarding local task.", e);
        } catch (IllegalTaskStateChangeException ignored) {
            // we don't care as we will only continue with the task from the storage provider.
        }
    }
}
