package cn.boboweike.carrot.server.dashboard;

import cn.boboweike.carrot.server.dashboard.mappers.CpuAllocationIrregularityNotificationMapper;
import cn.boboweike.carrot.server.dashboard.mappers.DashboardNotificationMapper;
import cn.boboweike.carrot.server.dashboard.mappers.NewCarrotVersionNotificationMapper;
import cn.boboweike.carrot.server.dashboard.mappers.SevereCarrotExceptionNotificationMapper;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;

import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.newInstance;
import static java.util.Arrays.asList;

public class DashboardNotificationManager {
    private final PartitionedStorageProvider storageProvider;
    private final Set<DashboardNotificationMapper<?>> notificationMappers;

    public DashboardNotificationManager(UUID backgroundTaskServerId, PartitionedStorageProvider storageProvider) {
        this.storageProvider = storageProvider;
        this.notificationMappers = new HashSet<>(asList(
                new SevereCarrotExceptionNotificationMapper(backgroundTaskServerId, storageProvider),
                new CpuAllocationIrregularityNotificationMapper(backgroundTaskServerId),
                new NewCarrotVersionNotificationMapper()
        ));
    }

    public void handle(Exception e) {
        if (e instanceof DashboardNotification) {
            notify((DashboardNotification) e);
        }
    }

    public void notify(DashboardNotification e) {
        notificationMappers.stream()
                .filter(notificationMapper -> notificationMapper.supports(e))
                .map(notificationMapper -> notificationMapper.map(e))
                .forEach(storageProvider::saveMetadata);
    }

    public void deleteNotification(Class<? extends DashboardNotification> notificationToDelete) {
        storageProvider.deleteMetadata(notificationToDelete.getSimpleName());
    }

    public <T extends DashboardNotification> T getDashboardNotification(Class<T> notificationClass) {
        return storageProvider
                .getMetadata(notificationClass.getSimpleName())
                .stream()
                .map(metadata -> newInstance(notificationClass, metadata))
                .findFirst()
                .orElse(null);
    }
}
