package cn.boboweike.carrot.server.dashboard.mappers;

import cn.boboweike.carrot.SevereCarrotException;
import cn.boboweike.carrot.configuration.Carrot;
import cn.boboweike.carrot.server.dashboard.DashboardNotification;
import cn.boboweike.carrot.storage.CarrotMetadata;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.storage.ThreadSafePartitionedStorageProvider;
import cn.boboweike.carrot.utils.RuntimeUtils;
import cn.boboweike.carrot.utils.metadata.VersionRetriever;

import java.time.Instant;
import java.util.UUID;

import static cn.boboweike.carrot.utils.StringUtils.isNotNullOrEmpty;
import static cn.boboweike.carrot.utils.diagnostics.DiagnosticsBuilder.diagnostics;
import static java.lang.String.format;

public class SevereCarrotExceptionNotificationMapper implements DashboardNotificationMapper<SevereCarrotException> {
    private final String id;
    private final PartitionedStorageProvider storageProvider;

    public SevereCarrotExceptionNotificationMapper(UUID backgroundTaskServerId, PartitionedStorageProvider storageProvider) {
        this.id = "BackgroundTaskServer " + backgroundTaskServerId;
        this.storageProvider = storageProvider;
    }

    @Override
    public boolean supports(DashboardNotification exception) {
        return exception instanceof SevereCarrotException;
    }

    @Override
    public CarrotMetadata mapToMetadata(SevereCarrotException notification) {
        String diagnosticsInfo = diagnostics()
                .withTitle(notification.getClass().getSimpleName() + " occurred in " + id + (isNotNullOrEmpty(notification.getMessage()) ? ": " + notification.getMessage() : ""))
                .withEmptyLine()
                .withSubTitle("Runtime information")
                .withBulletedLine("Timestamp", Instant.now().toString())
                .withBulletedLine("Location", id)
                .withBulletedLine("Carrot Version", VersionRetriever.getVersion(Carrot.class))
                .withBulletedLine("StorageProvider", storageProvider instanceof ThreadSafePartitionedStorageProvider ? ((ThreadSafePartitionedStorageProvider) storageProvider).getStorageProvider().getClass().getName() : storageProvider.getClass().getName())
                .withBulletedLine("Java Version", System.getProperty("java.version"))
                .withBulletedLine("Is running from nested jar", Boolean.toString(RuntimeUtils.isRunningFromNestedJar()))
                .withEmptyLine()
                .withSubTitle("Background Task Servers")
                .with(storageProvider.getBackgroundTaskServers(), (server, diagnosticsBuilder) -> diagnosticsBuilder.withBulletedLine(format("BackgroundTaskServer id: %s\n(workerPoolSize: %d, pollIntervalInSeconds: %d, firstHeartbeat: %s, lastHeartbeat: %s)", server.getId(), server.getWorkerPoolSize(), server.getPollIntervalInSeconds(), server.getFirstHeartbeat(), server.getLastHeartbeat())))
                .withEmptyLine()
                .withSubTitle("Diagnostics from exception")
                .withDiagnostics(2, notification.getDiagnostics())
                .withEmptyLine()
                .withSubTitle("Exception")
                .withException(notification)
                .asMarkDown();

        return new CarrotMetadata(SevereCarrotException.class.getSimpleName(), id, diagnosticsInfo);
    }
}
