package cn.boboweike.carrot.server.metrics;

import cn.boboweike.carrot.server.BackgroundTaskServer;
import io.micrometer.core.instrument.FunctionCounter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;

import java.util.ArrayList;
import java.util.List;
import java.util.function.ToDoubleFunction;

public class BackgroundTaskServerMetricsBinder implements AutoCloseable {
    private final BackgroundTaskServer backgroundTaskServer;
    private final MeterRegistry meterRegistry;
    private final List<Meter> meters;

    public BackgroundTaskServerMetricsBinder(BackgroundTaskServer backgroundTaskServer, MeterRegistry meterRegistry) {
        this.backgroundTaskServer = backgroundTaskServer;
        this.meterRegistry = meterRegistry;
        this.meters = new ArrayList<>();
        registerBackgroundTaskServerMetrics();
    }

    public void registerBackgroundTaskServerMetrics() {
        meters.add(registerFunction("poll-interval-in-seconds", bgTaskServer -> (double) bgTaskServer.getServerStatus().getPollIntervalInSeconds()));
        meters.add(registerFunction("worker-pool-size", bgTaskServer -> (double) bgTaskServer.getServerStatus().getWorkerPoolSize()));

        meters.add(registerGauge("process-all-located-memory", bgTaskServer -> (double) bgTaskServer.getServerStatus().getProcessAllocatedMemory()));
        meters.add(registerGauge("process-free-memory", bgTaskServer -> (double) bgTaskServer.getServerStatus().getProcessFreeMemory()));
        meters.add(registerGauge("system-free-memory", bgTaskServer -> (double) bgTaskServer.getServerStatus().getSystemFreeMemory()));
        meters.add(registerGauge("system-total-memory", bgTaskServer -> (double) bgTaskServer.getServerStatus().getSystemTotalMemory()));
        meters.add(registerGauge("first-heartbeat", bgTaskServer -> (double) bgTaskServer.getServerStatus().getFirstHeartbeat().getEpochSecond()));
        meters.add(registerGauge("last-heartbeat", bgTaskServer -> (double) bgTaskServer.getServerStatus().getLastHeartbeat().getNano()));
    }

    private FunctionCounter registerFunction(String name, ToDoubleFunction<BackgroundTaskServer> func) {
        return FunctionCounter.builder(toMicroMeterName(name), this.backgroundTaskServer, func).tag("id", this.backgroundTaskServer.getId().toString()).register(meterRegistry);
    }

    private Gauge registerGauge(String name, ToDoubleFunction<BackgroundTaskServer> func) {
        return Gauge.builder(toMicroMeterName(name), this.backgroundTaskServer, func).tag("id", this.backgroundTaskServer.getId().toString()).register(meterRegistry);
    }

    private String toMicroMeterName(String name) {
        return "carrot.background-task-server." + name;
    }

    @Override
    public void close() {
        meters.forEach(meter -> {
            try {
                meter.close();
            } catch (Exception e) {
                // nothing more we can do
            }
        });
    }
}
