package cn.boboweike.carrot.server.runner;

import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskParameter;
import cn.boboweike.carrot.tasks.context.TaskContext;
import cn.boboweike.carrot.tasks.TaskDetails;
import cn.boboweike.carrot.utils.TaskUtils;

import java.lang.reflect.Method;
import java.util.List;
import java.util.stream.IntStream;

import static cn.boboweike.carrot.utils.reflection.ReflectionUtils.newInstance;

public abstract class AbstractBackgroundTaskRunner implements BackgroundTaskRunner {

    protected abstract BackgroundTaskWorker getBackgroundTaskWorker(Task task);

    public void run(Task task) throws Exception {
        getBackgroundTaskWorker(task).run();
        if (Thread.currentThread().isInterrupted()) throw new InterruptedException();
    }

    protected static class BackgroundTaskWorker {

        protected final Task task;
        protected final TaskDetails taskDetails;

        public BackgroundTaskWorker(Task task) {
            this.task = task;
            this.taskDetails = task.getTaskDetails();
        }

        public void run() throws Exception {
            Class<?> taskToPerformClass = getTaskToPerformClass();
            Object taskToPerform = getTaskToPerform(taskToPerformClass);
            Method taskMethodToPerform = getTaskMethodToPerform(taskToPerformClass);
            invokeTaskMethod(taskToPerform, taskMethodToPerform);
        }

        protected Class<?> getTaskToPerformClass() {
            return TaskUtils.getTaskClass(taskDetails);
        }

        protected Object getTaskToPerform(Class<?> taskToPerformClass) {
            return newInstance(taskToPerformClass);
        }

        protected Method getTaskMethodToPerform(Class<?> taskToPerformClass) {
            return TaskUtils.getTaskMethod(taskToPerformClass, taskDetails);
        }

        protected void invokeTaskMethod(Object taskToPerform, Method taskMethodToPerform) throws Exception {
            final Object[] taskParameterValues = taskDetails.getTaskParameterValues();
            final List<TaskParameter> taskParameters = taskDetails.getTaskParameters();

            IntStream.range(0, taskParameters.size())
                    .filter(i -> taskParameters.get(i).getClassName().equals(TaskContext.class.getName()))
                    .findFirst()
                    .ifPresent(index -> taskParameterValues[index] = getRunnerTaskContext());

            try {
                ThreadLocalTaskContext.setTaskContext(getRunnerTaskContext());
                taskMethodToPerform.invoke(taskToPerform, taskParameterValues);
            } finally {
                ThreadLocalTaskContext.clear();
            }
        }

        protected RunnerTaskContext getRunnerTaskContext() {
            return new RunnerTaskContext(task);
        }
    }
}
