package cn.boboweike.carrot.server.tasks;

import cn.boboweike.carrot.server.BackgroundTaskServer;
import cn.boboweike.carrot.storage.PartitionedStorageProvider;
import cn.boboweike.carrot.tasks.AbstractTask;
import cn.boboweike.carrot.tasks.states.StateName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Set;
import java.util.stream.Collectors;

import static cn.boboweike.carrot.CarrotException.shouldNotHappenException;
import static cn.boboweike.carrot.utils.CollectionUtils.asSet;
import static cn.boboweike.carrot.utils.TaskUtils.taskExists;
import static java.util.stream.Collectors.toSet;

public class CheckIfAllTasksExistTask implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackgroundTaskServer.class);

    private final PartitionedStorageProvider storageProvider;

    public CheckIfAllTasksExistTask(BackgroundTaskServer backgroundTaskServer) {
        this.storageProvider = backgroundTaskServer.getStorageProvider();
    }

    @Override
    public void run() {
        try {
            final Set<String> distinctRecurringTaskSignatures = getDistinctRecurringTaskSignaturesThatDoNotExistAnymore();
            final Set<String> distinctScheduledTaskSignatures = getDistinctScheduledTaskSignaturesThatDoNotExistAnymore();

            Set<String> tasksThatCannotBeFound = asSet(distinctRecurringTaskSignatures, distinctScheduledTaskSignatures);

            if (!distinctRecurringTaskSignatures.isEmpty() || !distinctScheduledTaskSignatures.isEmpty()) {
                String taskStateThatIsNotFound = taskTypeNotFoundLabel(distinctRecurringTaskSignatures, distinctScheduledTaskSignatures);
                LOGGER.warn("Carrot found {} tasks that do not exist anymore in your code. These tasks will fail with a TaskNotFoundException (due to a ClassNotFoundException or a MethodNotFoundException)." +
                                "\n\tBelow you can find the method signatures of the tasks that cannot be found anymore: {}",
                        taskStateThatIsNotFound,
                        tasksThatCannotBeFound.stream().map(sign -> "\n\t" + sign + ",").collect(Collectors.joining())
                );
            }
        } catch (Exception e) {
            LOGGER.error("Unexpected exception running `CheckIfAllTasksExistTask`", shouldNotHappenException(e));
        }
    }

    private Set<String> getDistinctRecurringTaskSignaturesThatDoNotExistAnymore() {
        return storageProvider.getRecurringTasks().stream()
                .map(AbstractTask::getTaskSignature)
                .filter(taskSignature -> !taskExists(taskSignature))
                .collect(toSet());
    }

    private Set<String> getDistinctScheduledTaskSignaturesThatDoNotExistAnymore() {
        return storageProvider.getDistinctTaskSignatures(StateName.SCHEDULED).stream()
                .filter(taskSignature -> !taskExists(taskSignature))
                .collect(toSet());
    }

    private String taskTypeNotFoundLabel(Set<String> distinctRecurringTaskSignaturesThatDoNotExistAnymoreAfterCleanup, Set<String> distinctScheduledTaskSignaturesThatDoNotExistAnymoreAfterCleanup) {
        String taskStateThatIsNotFound = "";
        if (!distinctRecurringTaskSignaturesThatDoNotExistAnymoreAfterCleanup.isEmpty()) {
            taskStateThatIsNotFound += "RECURRING";
        }
        if (!distinctRecurringTaskSignaturesThatDoNotExistAnymoreAfterCleanup.isEmpty() && !distinctScheduledTaskSignaturesThatDoNotExistAnymoreAfterCleanup.isEmpty()) {
            taskStateThatIsNotFound += " AND ";
        }
        if (!distinctScheduledTaskSignaturesThatDoNotExistAnymoreAfterCleanup.isEmpty()) {
            taskStateThatIsNotFound += "SCHEDULED";
        }
        return taskStateThatIsNotFound;
    }
}
