package cn.boboweike.carrot.storage;

import cn.boboweike.carrot.storage.listeners.*;
import cn.boboweike.carrot.tasks.Task;
import cn.boboweike.carrot.tasks.TaskId;
import cn.boboweike.carrot.utils.resilience.RateLimiter;
import cn.boboweike.carrot.utils.streams.StreamUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

public abstract class AbstractPartitionedStorageProvider implements PartitionedStorageProvider, AutoCloseable {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractPartitionedStorageProvider.class);

    private final Set<StorageProviderChangeListener> onChangeListeners;
    private final TaskStatsEnricher taskStatsEnricher;
    private final RateLimiter changeListenerNotificationRateLimit;
    private final ReentrantLock reentrantLock;
    private volatile Timer timer;

    protected AbstractPartitionedStorageProvider(RateLimiter changeListenerNotificationRateLimit) {
        this.onChangeListeners = ConcurrentHashMap.newKeySet();
        this.taskStatsEnricher = new TaskStatsEnricher();
        this.changeListenerNotificationRateLimit = changeListenerNotificationRateLimit;
        this.reentrantLock = new ReentrantLock();
    }

    @Override
    public String getName() {
        return this.getClass().getSimpleName();
    }

    @Override
    public void addTaskStorageOnChangeListener(StorageProviderChangeListener listener) {
        onChangeListeners.add(listener);
        startTimerToSendUpdates();
    }

    @Override
    public void removeTaskStorageOnChangeListener(StorageProviderChangeListener listener) {
        onChangeListeners.remove(listener);
        if (onChangeListeners.isEmpty()) {
            stopTimerToSendUpdates();
        }
    }

    @Override
    public void close() {
        stopTimerToSendUpdates();
    }

    protected void notifyTaskStatsOnChangeListenersIf(boolean mustNotify) {
        if (mustNotify) {
            notifyTaskStatsOnChangeListeners();
        }
    }

    protected void notifyTaskStatsOnChangeListeners() {
        try {
            if (changeListenerNotificationRateLimit.isRateLimited()) return;

            final List<TaskStatsChangeListener> taskStatsChangeListeners = StreamUtils
                    .ofType(onChangeListeners, TaskStatsChangeListener.class)
                    .collect(toList());
            if (!taskStatsChangeListeners.isEmpty()) {
                TaskStatsData taskStatsData = getTaskStatsData();
                TaskStats taskStats = taskStatsData.getOverallTaskStats();
                TaskStatsExtended taskStatsExtended = taskStatsEnricher.enrich(taskStats);
                taskStatsData.setOverallTaskStats(taskStatsExtended);
                taskStatsChangeListeners.forEach(listener -> listener.onChange(taskStatsData));
            }
        } catch (Exception e) {
            logError(e);
        }
    }

    protected void notifyMetadataChangeListeners(boolean mustNotify) {
        if (mustNotify) {
            notifyMetadataChangeListeners();
        }
    }

    protected void notifyMetadataChangeListeners() {
        try {
            final Map<String, List<MetadataChangeListener>> metadataChangeListenersByName = StreamUtils
                    .ofType(onChangeListeners, MetadataChangeListener.class)
                    .collect(groupingBy(MetadataChangeListener::listenForChangesOfMetadataName));

            if (!metadataChangeListenersByName.isEmpty()) {
                metadataChangeListenersByName.forEach((metadataName, listeners) -> {
                    List<CarrotMetadata> carrotMetadata = getMetadata(metadataName);
                    listeners.forEach(listener -> listener.onChange(carrotMetadata));
                });
            }
        } catch (Exception e) {
            logError(e);
        }
    }

    private void notifyTaskChangeListeners() {
        try {
            final Map<TaskId, List<TaskChangeListener>> listenerByTask = StreamUtils
                    .ofType(onChangeListeners, TaskChangeListener.class)
                    .collect(groupingBy(TaskChangeListener::getTaskId));
            if (!listenerByTask.isEmpty()) {
                listenerByTask.forEach((taskId, listeners) -> {
                    try {
                        Task task = getTaskById(taskId);
                        listeners.forEach(listener -> listener.onChange(task));
                    } catch (TaskNotFoundException taskNotFoundException) {
                        // somebody is listening for a Task that does not exist
                        listeners.forEach(taskChangeListener -> {
                            try {
                                taskChangeListener.close();
                            } catch (Exception e) {
                                // Not relevant?
                            }
                        });
                    }
                });
            }
        } catch (Exception e) {
            logError(e);
        }
    }

    private void notifyBackgroundTaskServerStatusChangeListeners() {
        try {
            final List<BackgroundTaskServerStatusChangeListener> serverChangeListeners = StreamUtils
                    .ofType(onChangeListeners, BackgroundTaskServerStatusChangeListener.class)
                    .collect(toList());
            if (!serverChangeListeners.isEmpty()) {
                List<BackgroundTaskServerStatus> servers = getBackgroundTaskServers();
                serverChangeListeners.forEach(listener -> listener.onChange(servers));
            }
        } catch (Exception e) {
            logError(e);
        }
    }

    void startTimerToSendUpdates() {
        if (timer == null) {
            try {
                if (reentrantLock.tryLock()) {
                    timer = new Timer(true);
                    timer.schedule(new NotifyOnChangeListeners(), 3000, 5000);
                }
            } finally {
                reentrantLock.unlock();
            }
        }
    }

    void stopTimerToSendUpdates() {
        if (timer != null) {
            boolean canCancelTimer = timer != null && reentrantLock.tryLock();
            if (canCancelTimer) {
                timer.cancel();
                timer = null;
                reentrantLock.unlock();
            }
        }
    }

    private void logError(Exception e) {
        if (reentrantLock.isLocked() || timer == null) return; // timer is being stopped so not interested in it
        LOGGER.warn("Error notifying TaskStorageChangeListeners", e);
    }

    class NotifyOnChangeListeners extends TimerTask {

        public void run() {
            notifyTaskStatsOnChangeListeners();
            notifyTaskChangeListeners();
            notifyBackgroundTaskServerStatusChangeListeners();
            notifyMetadataChangeListeners();
        }
    }
}
